% FBP_CPU  wrapper around iradonfast_v3 that automatically splits the data into
% smaller blocks to avoid too large memory allocation and emulate GPU based
% function tomo.FBP
%
% tomogram = FBP_CPU(sinogram, cfg, vectors, varargin)
% 
% Inputs:
%     **sino        - sinogram (Nlayers x width x Nangles)
%     **cfg         - config struct from ASTRA_initialize
%     **vectors     - vectors of projection rotation generated by ASTRA_initialize
%  *optional*
%     ** valid_angles =  []       - list of valid angles, []==all are valid
%     ** filter =   'ram-lak'     - name of the FBP filter 
%     ** filter_value =   1       - fitlering value for the FBP filter 
%     ** verbose =  1             - verbose = 0 : quiet, verbose : standard info , verbose = 2: debug 
%     ** use_derivative =  false  - calculate reconstruction from the phase derivative 
% *returns*
%     ++tomogram    - FBP reconstruction 

%*-----------------------------------------------------------------------*
%|                                                                       |
%|  Except where otherwise noted, this work is licensed under a          |
%|  Creative Commons Attribution-NonCommercial-ShareAlike 4.0            |
%|  International (CC BY-NC-SA 4.0) license.                             |
%|                                                                       |
%|  Copyright (c) 2017 by Paul Scherrer Institute (http://www.psi.ch)    |
%|                                                                       |
%|       Author: CXS group, PSI                                          |
%*-----------------------------------------------------------------------*
% You may use this code with the following provisions:
%
% If the code is fully or partially redistributed, or rewritten in another
%   computing language this notice should be included in the redistribution.
%
% If this code, or subfunctions or parts of it, is used for research in a 
%   publication or if it is fully or partially rewritten for another 
%   computing language the authors and institution should be acknowledged 
%   in written form in the publication: “Data processing was carried out 
%   using the “cSAXS matlab package” developed by the CXS group,
%   Paul Scherrer Institut, Switzerland.” 
%   Variations on the latter text can be incorporated upon discussion with 
%   the CXS group if needed to more specifically reflect the use of the package 
%   for the published work.
%
% A publication that focuses on describing features, or parameters, that
%    are already existing in the code should be first discussed with the
%    authors.
%   
% This code and subroutines are part of a continuous development, they 
%    are provided “as they are” without guarantees or liability on part
%    of PSI or the authors. It is the user responsibility to ensure its 
%    proper use and the correctness of the results.

function tomogram = FBP_CPU(sinogram, cfg, vectors, varargin)

    par = inputParser;
    par.addParameter('valid_angles', [])
    par.addParameter('filter',  'ram-lak' )
    par.addParameter('filter_value',  1 )
    par.addOptional('verbose', 1)   % verbose = 0 : quiet, verbose : standard info , verbose = 2: debug 
    par.addOptional('determine_weights', true)% reweight projections if the angles are not equidistant
    par.addOptional('mask', [])   % apply mask on reconstruction 
    par.addOptional('use_derivative', false)   % calculate reconstruction from the phase derivative 
    par.KeepUnmatched = true; 
    par.parse(varargin{:})
    r = par.Results;
    
    if r.verbose
        disp('====== FBP ==========')
    end
    if r.use_derivative
        extra_args = {'derivative'};
    else
        extra_args = {};
    end

    
    if ~isempty(r.valid_angles)
        sinogram = sinogram(:,:,r.valid_angles);
        vectors = vectors(r.valid_angles,:);
    end
    
    % calculate the original angles 
    theta = rad2deg(pi-atan2(vectors(:,2),-vectors(:,1)))'; 
    lamino_angle = rad2deg(pi/2-atan2(vectors(:,3), vectors(:,1)./cos(theta))); 
    if abs(lamino_angle - 90) > 1e-2
       error('CPU implementation of FBP does not support laminography') 
    end
    
    [Nlayers,tomo_size,Nangles] = size(sinogram) ; 
    block_size = ceil(2e9/(tomo_size*Nangles*4)); % split the task into ~2GB blocks 
    Nblocks = ceil(Nlayers / block_size ); 

    % preallocate array to store results 
    tomogram = zeros(tomo_size, tomo_size, Nlayers, 'single');
    
    for ii = 1:Nblocks
        ind = 1+(ii-1)*block_size:min(ii*block_size, Nlayers);
        tomogram(:,:,ind) =tomo.iradonfast_v3(permute(sinogram(ind,:,:),[2,3,1]), theta, 'linear', r.filter, tomo_size, r.filter_value, extra_args{:});   % Calculate slice
    end
    
    % make the results equivalent to the GPU version
    tomogram = utils.imshift_fft(tomogram, [0.5,0.5] ); 
    tomogram = utils.crop_pad(tomogram, [cfg.iVolX, cfg.iVolY]);
    
    if ~isempty(r.mask)
        tomogram = tomogram .* r.mask;
    end
    
end
