% template_tomo_recons.m
% to process ptychographic reconstructions from many projections
% and make local tomographic reconstruction using a low resolution tomogram

% This code and subroutines are part of a continuous develpment. There is no 
% liability on PSI or cSAXS. Before publishing results using this code 
% please notify M. Guizar-Sicairos, (mguizar@gmail.com)
% or another PSI cSAXS staff member.

% Publications most relevant to this code
% M. Guizar-Sicairos, A. Diaz, M. Holler, M. S. Lucas, A. Menzel, R. A. Wepf, and O. Bunk
% "Phase tomography from x-ray coherent diffractive imaging projections," Opt. Express 19, 21345-21357 (2011).
% For tomographic consistency alignment:
% M. Guizar-Sicairos, J. J. Boon, K. Mader, A. Diaz, A. Menzel, and O. Bunk,
% "Quantitative interior x-ray nanotomography by a hybrid imaging technique," Optica 2, 259-266 (2015).



close all 

%% Add paths
base_path='../../';
addpath('utils')
addpath('tests')
addpath(find_base_package())



import plotting.*
import io.*
if length(dbstack) == 1  
    debug(false) % keep debug option false in case of manual run 
end
if debug() <= 1  % control point for the automatic code tests


%% File management
%%%%%%%%%%%%%%%%%%%%%%%%%
%%% Edit this section %%%
%%%%%%%%%%%%%%%%%%%%%%%%%

par.scanstomo = [];
par.tomo_id = 139; % Either scan numbers or tomo_id can be given, but not both, if not provided leave tomo_id=[]

% path to stored low resolution delta_tomogram 
par.lowres_tomo_path = '../../analysis_tomo/tomo_id_67_S00089_to_S01525_test_1_recons_/tomogram_delta_S00089_to_S01525_ram-lak_freqscl_1.00.mat'; 

% verbosity and online tomo 
par.online_tomo = length(dbstack)==3;   % automatically run if called from externally
par.verbose_level = 1; 
par.base_path=base_path;
par.showrecons = par.verbose_level > 1;

% IO loading 
par.fileprefix='';                  % string at the beginning of the filename, related to reconstruction name
par.filesuffix = '_recons';   % string searched at the end of the data filenames,  No need to add _c or _recons, it will look for it 
par.file_extension = 'h5';
par.downsample_projections = 0;     % downsample projections by factor of 2^x, set 0 to do nothing and 1,2,.. for different levels of projection binning 

par.use_mex_loader = true;   % if true, use the new fast MEX-based loader, false is rather a backup option
par.Nthreads_mexread = 10;   % number of threads used by the fast MEX based data loader
par.analysis_path = [par.base_path 'analysis/'];      % path from where projections are loaded 
par.output_path = [par.base_path 'analysis_tomo/'];   % path where the tomograms are stored 
par.clip_amplitude_quantile = 0.95;  %  clip amplitudes in the loaded projections that are exceeding given quantile, if par.clip_amplitude_quantile == 1, do nothing 
par.max_residua_limit = 100;        % limit used to determine which projection have failed 
par.sort_by_angle = false;          % If true projections are sorted by angle rather than by the scan numbers, this sorting can effect alignment 



% IO saving
par.output_folder_prefix = '';      % a string that will be attached to the output folder name in form tomo_+output_folder_prefix+autogenerated_description
par.save_final_data = true; 
par.force_overwrite = par.online_tomo ;     % ask before overwritting data 
par.save_temp_data = false;      % 1 to save temporal data to disk, ! final results are saved always !

% Angles + auto colibration 
par.use_OMNY_file_angles = 1;        %% Read angles from special file in OMNY, otherwise it uses SPEC dat file
par.checkangles = false;             %% Compares spec angles with expected
par.auto_alignment = 0;     %% Uses interferometer positions to prealign projections - for now it will read them from the spec file. 
			% This feature needs a new calibration of the sphere and is not fully tested.
par.get_auto_calibration = 1;  % If you want to use this recons to get a calibration file for autotomo  
par.remove_duplicated_angles = true; % remove angles measured twice, should be true for FBP method 
par.surface_calib_file = [par.base_path 'matlab/tomo/position_calibration_20181026.mat'];
par.omnyposfile = [par.base_path 'specES1/scan_positions/scan_%05d.dat'];		%Filename pattern for Orchestra interferometer position files   
par.OMNY_angle_file = [par.base_path, 'specES1/dat-files/tomography_scannumbers.txt'];  % Filename with angles

% Other
par.showsorted = true;      % sort plotted projections by angles, it has not effect on the alignment itself 
par.windowautopos = true;
par.remove_duplicated_angles = true; % remove angles measured twice, should be true for FBP method 
par.save_memory = false;        % try to limit use of RAM 
    par.inplace_processing = par.save_memory; % process object_stack using inplace operations to save memory 
    par.fp16_precision     = true; % use 16-bit precision to store the complex-valued projections 
    par.cache_stack_object = par.save_memory; % store stack_object to disk when no needed 



par.GPU_list = [1];     % number of the used GPU % If you want to check usage of GPU 
                        % > nvidia-smi
                        % Then in matlab use  par.GPU_list = 2  for example to use the second GPU 
par.Nworkers = min(10,feature('numcores'));  % number of workers for parfor. avoid starting too many workers

%%% reconstruction settings 

% Geometry settings, important for laminography 
par.is_laminography = false;        % false -> standard tomography, true = allow some specific options for laminography reconstruction, ie circular field of view ... 
par.is_interior_tomo = true;         % force preferences for interior tomography

%%%% geometry settings %% 
par.lamino_angle = 90;              % laminography angle, for normal tomo set 90
par.tilt_angle = 0;                 % rotation of the camera around the beam direction 
par.vertical_scale = 1;             % relative pixel scale between the projection and reconstruction 
par.horizontal_scale = 1;           % relative pixel scale between the projection and reconstruction 
%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%

block_fun_cfg = struct('GPU_list', par.GPU_list, 'inplace', par.inplace_processing);   % instruction for stack_object processing
object_preprocess_fun = @(x)x;    % data preprocessing function, useful for laminography

end
if debug() == 1 % control point for the automatic code tests
    return
end


%%% load angles for the projections 
[par,theta] = prepare.load_angles(par, par.scanstomo, par.tomo_id);


%% Reads ptychographic reconstructions and stores them in stack_object
%%%%%%%%%%%%%%%%%%%%%%%%%
%%% Edit this section %%%
%%%%%%%%%%%%%%%%%%%%%%%%%
exclude_scans = [];
%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%


%%% perform initial checks 
[par, angles_check] = prepare.initialize_tomo(par, par.scanstomo, true); 


[stack_object, theta, ~, par] = prepare.load_projections_fast(par,exclude_scans,par.dims_ob,theta,object_preprocess_fun);



%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%% %%%%%%%%%%%% ALIGN AND RECONSTRUCTION INTERIOR TOMOGRAPHY %%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%


%% Cross correlation with the low resolution model -> get initial alignment of the interior projections 
% + use low resolution model to remove phase ramp from stack_object
[stack_object, tomogram_lowres, weight, total_shift, par.resolution_ratio, Nw_lres] = ...
    interior.align_projections_to_lowres_tomogram(stack_object, par.lowres_tomo_path, theta, par);


%% Display reconstructed phases
%%%%%%%%%%%%%%%%%%%%%%%%%
%%% Edit this section %%%
%%%%%%%%%%%%%%%%%%%%%%%%%
par.baraxis = 'auto';       % = 'auto'  or   = [-1 1]
par.windowautopos = true;  % automatic placemement of the plot
par.showsorted = true;      % sort the projections by angles 
%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%
[Nl_s, Nw_s, ~] = size(stack_object);
object_ROI_local = {ceil(1+par.asize(1)/2:Nl_s-par.asize(1)/2),ceil(1+par.asize(2)/2:Nw_s-par.asize(2)/2)}; 

tomo.show_projections(stack_object, theta, par, 'fnct', @angle, ...
    'title', 'Full original projections before alignmnent','plot_residua', false, ...
    'rectangle_pos', [object_ROI_local{2}(1), object_ROI_local{2}(end), object_ROI_local{1}(1), object_ROI_local{1}(end)]) 





total_shift_orig = total_shift; 

Nangles = length(theta); 

Npix_lres = size(tomogram_lowres);
    
% size of the low resolution projections to be generated 
Nl = ceil(Nw_lres(1)*par.resolution_ratio); % get the corresponding size of the low res tomogram would be measured in full resolution 
Nw = ceil(Nw_lres(2)*par.resolution_ratio); 

% size of air region in the full size reconstruction 
par.air_gap = ceil([Nw,Nw]/40);    % very roughly distance around sides where is assumed air, it is used only for initial guess       

        

% Make data easily splitable for ASTRA
object_ROI = {1:Nl, 1:Nw}; 
width_sinogram = floor(length(object_ROI{2})/32)*32;
Nlayers = floor(length(object_ROI{1})/32)*32;
Nroi = [length(object_ROI{1}),length(object_ROI{2})];
object_ROI = {object_ROI{1}(ceil(Nroi(1)/2))+[1-Nlayers/2:Nlayers/2],...
              object_ROI{2}(ceil(Nroi(2)/2))+[1-width_sinogram/2:width_sinogram/2]}; 


          
%% TOMOCONSISTENCY ALIGNMENT
%%%%%%%%%%%%%%%%%%%%%%%%%
%%% Edit this section %%%
%%%%%%%%%%%%%%%%%%%%%%%%%

%%%%%%%%%% RECONSTRUCTION PARAMETERS %%%%%%%%%%%%%%%%%%%%
Npix = ceil(width_sinogram/32)*32;  % for pillar it can be the same as width_sinogram
par.vert_range = ceil(Nl/2+mean(total_shift(:,2)) + [-(Nl_s/2-par.asize(1)/2):(Nl_s/2-par.asize(1)/2)]); % selected vertical layers for alignment 

%%%%%%%%%% Other parameters %%%%%%%%
max_binning = 2^ceil(log2(min([Npix, length(par.vert_range)]))-log2(20));   % prevent binning to make smaller object than 20 pixels, reconstruction gets unstable 
min_binning = min([max_binning, 2^max(2*(debug()>0), par.online_tomo)]) ; 

par.high_pass_filter = 0.01;        % remove effect of residuums/bad ptycho convergence , high value may get stack in local minima , low value increases sensitivity to residua / phase jumps 
par.showsorted = true;              % selected angular order for alignment 
par.valid_angles = 1:Nangles > 0;   % use only this indices to do reconstruction (with respect to the time sorted angles)
par.align_horizontal = true;        % horizontal alignment 
par.align_vertical = true;          % vertical alignment, usually only a small correction of initial guess  
par.use_mask = false;               % apply support mask on the reconstructed sample volume 
par.mask_threshold = 0.001;         % []== Otsu thresholding, otherwise the value is in phase tomogram reconstruction 
par.use_localTV = false;            % apply local TV into the reconstructed volume, should help with convergence  
par.apply_positivity = true;        % remove negative values in the reconstructed volume, helps with alignment especially for sparser samples 
par.min_step_size  = 0.01;          % stoppig criterion ( subpixel precision )
par.max_iter = 500;                 % maximal number of iterations
par.use_Xcorr_outlier_check = true; % in the first iteration check and remove outliers using Xcorr 

%%%  Internal parameters, do not change %%%%%%%%
par.step_relaxation = 0.3;          % gradient decent step relaxation, (1 == full step), value <1 may be needed to avoid oscilations  
par.filter_type = 'ram-lak';        % FBP filter (ram-lak, hamming, ....)
par.freq_scale = 1;                 % Frequency cutoff for the FBP filters, 1 == no FBP filtering
par.refine_geometry = false;        % try to find better geoometry to fit the data

%%%%% DO NO EDIT %%%%%%%%

par.unwrap_data_method = 'none';
par.exterior_weight = 0.05;          % relative weight of the exterion region. It should be > 0 to avoid drifts of the local tomo and << 1 to avoid 
par.momentum_acceleration = false;   % accelerate convergence by momentum gradient descent method 
par.center_reconstruction = false;   % keep the center of mass of the reconstructed sample in center of 3D field of view
%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%

binning = 2.^(log2(max_binning):-1:log2(min_binning)); 
shift = zeros(Nangles,2); 

gpu = gpuDevice;
utils.verbose(-1,'Alignment')

for jj = 1:length(binning)
    par.binning = binning(jj);
    utils.verbose(-1, 'Binning %i', par.binning)
        
    disp('Apply shift corrections on the interior dataset + merge with low-resolution tomogram ')
    clear sinogram sino_weights 


    % merge interior and low resolution projections, the low resolution one is already aligned !! , no shifts should be needed 
    Nblocks =  ceil( (16*6*4* length(par.vert_range) *Nw*Nangles) /  gpu.AvailableMemory) ; 
    [sinogram, sino_weights] = tomo.block_fun(@interior.merge_and_unwrap_sinograms,stack_object, gpuArray(tomogram_lowres), weight, total_shift,reshape(theta,1,1,[]),object_ROI, Nw_lres,par, struct('use_fp16', true, 'Nblocks', Nblocks ));

    %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
    %% ALIGN BINNED SINOGRAMS
    %% provide the already binned sinograms to the tomo.align_tomo_consistency_linear and adjust the corresponding parameters here 
    %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
    par_tmp = par; 
    par_tmp.binning = 1;
    par_tmp.vert_range = []; 
    par_tmp.air_gap = par.air_gap/par.binning; 
    par_tmp.unwrap_data_method = 'fft_1d';
    % start acceleration only after the first iteration 
    par_tmp.momentum_acceleration = jj > 1;
    Npix_tmp = ceil(Npix/par.binning);

    shift = zeros(Nangles,2); 
    % self consitency based alignment procedure based on the ASTRA toolbox 
    [shift, ~, ~, err] = tomo.align_tomo_consistency_linear(sinogram,sino_weights, theta, Npix_tmp, shift/par.binning, par_tmp); 
    shift = shift .* par.binning;

    total_shift = total_shift + shift;

    %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
    %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

    % plot the estimated shifts 
    plotting.smart_figure(25)
    clf()
    plot(theta, total_shift_orig - total_shift, '.')
    legend({ 'Horizontal shift', 'Vertical shift'})
    xlabel('Angle')
    ylabel('Shift [px]')
    xlabel('Sorted angles')
    title('Additional shift correction from self-consistency alignment')
    axis tight ; grid on 
    drawnow

    
end
utils.verbose(-1,'Alignment finished')

clear sino sino_weights 


%%%%%%%%%%%%%%%%%%%%%%%%%%%
% End of alignment 
%%%%%%%%%%%%%%%%%%%%%%%%%%%


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%% Show movie with aligned images
%%%%%%%%%%%%%%%%%%%%%%%%%
%%% Edit this section %%%
%%%%%%%%%%%%%%%%%%%%%%%%%
par.baraxis = 'auto';       % = 'auto'  or   = [-1 1]
par.windowautopos = true;  % automatic placemement of the plot
par.showsorted = true;      % sort the projections by angles 
%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%


tomo.show_projections(sinogram, theta, par,...
    'title', 'Projections after self-consistent alignment')




%% interior tomogram - 1) do only interior part to check FSC
%%%%%%%%%%%%%%%%%%%%%%%%%
%%% Edit this section %%%
%%%%%%%%%%%%%%%%%%%%%%%%%
par.usecircle = true;               % Use circle to mask out the corners of the tomogram
par.filter_type = 'ram-lak';        % FBP filter (ram-lak, hamming, ....)
par.freq_scale = 1;                 % Frequency cutoff
rec_ind = find(par.valid_angles);   % use only some angles 
apodize = 0;                        % axial apodization 
radial_smooth_apodize =min(Npix)/50;% smooths the circle function (circulo), which delimits the reconstruction, this to avoid sharp edges in the  reconstruction
par.unwrap_data_method = 'fft_1d';    % phase unwrapping methods: fft_1d, none 
vert_range = 'interior';              % vertical region to be reconstructed, options:  'interior' , or 'full'
                                    %                                                [] - vert_range = object_ROI{1}
%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%
utils.verbose(struct('prefix', 'recons'))

if par.online_tomo || strcmpi(vert_range, 'interior') || debug()
    vert_range = ceil(Nl/2+mean(total_shift(:,2)) + [-(Nl_s/2-par.asize(1)/2):(Nl_s/2-par.asize(1)/2)]); % selected vertical layers for alignment 
    reconstruct_ROI = {vert_range, object_ROI{2}};
else
    reconstruct_ROI = object_ROI; 
end

circulo = [];  % apodization function 
par.vert_range = []; % dont change

utils.verbose(-1,'Generating sinograms ...')
% merge interior and low resolution projections, the low resolution one is already aligned !! , no shifts should be needed 
par.binning = 1; 
clear sinogram
Nblocks =  ceil( (4*6*4* Nl*Nw*Nangles) /  gpu.AvailableMemory) ; 
[sinogram, sino_weights] = tomo.block_fun(@interior.merge_and_unwrap_sinograms,stack_object, gpuArray(tomogram_lowres), weight, total_shift,reshape(theta,1,1,[]),reconstruct_ROI, Nw_lres, par, struct('use_fp16', true, 'Nblocks', Nblocks ));
utils.verbose(-1,'Sinograms done')

% Get two reconstructions for FSC   %%%%%%%%%%%%%%
[~,ind_sort] = sort(theta); 
ind_sort = ind_sort(ismember(ind_sort, rec_ind));
ind_rec = {ind_sort(1:2:end), ind_sort(2:2:end)}; 


% estimate Npix of the interior tomogram
Npix_small(1:2) = ceil(sum(any(weight>0.5*max(weight(:)),1)) .* Nl_s / size(weight,1)/4)*4; 
Npix_small(3) = ceil(sum(any(weight>0.5*max(weight(:)),2)) .* Nw_s / size(weight,2)/4)*4; 
Npix_small(3) = min(Npix_small(3), size(sinogram,1));

% select only sinogram region requited for the local tomography 
sinogram_local = utils.crop_pad(sinogram, min(size(sinogram), [Npix_small([3,1]),inf]));

[Nw_sino,Nl_sino,~] = size(sinogram_local);

% generate configuration
[cfg, vectors] = astra.ASTRA_initialize(Npix_small,[Nw_sino,Nl_sino],theta,par.lamino_angle,par.tilt_angle); 
% find optimal split of the dataset for given GPU 
split = astra.ASTRA_find_optimal_split(cfg, length(par.GPU_list), 2);
tomograms = {};
utils.verbose(-1,'Reconstructing ... ')
for ii = 1:2
    % FBP code 
    tomograms{ii} = tomo.FBP_zsplit(sinogram_local, cfg, vectors, split,'valid_angles',ind_rec{ii},...
        'GPU', par.GPU_list,'filter',par.filter_type, 'filter_value',par.freq_scale,...
        'use_derivative', strcmpi(par.unwrap_data_method, 'none'), 'mask', circulo, 'padding', 'symmetric');
end
utils.verbose(-1,'FBP reconstruction done')
clear sinogram_local


%% 3D FSC 
%%%%%%%%%%%%%%%%%%%%%%%%%
%%% Edit this section %%%
%%%%%%%%%%%%%%%%%%%%%%%%%
rad_apod = Npix_small(1)/5;           % Radial apodization
axial_apod = 20;         % Axial apodization
radial_smooth = (Npix_small(1)/2-rad_apod)/10; % smooths the circle function (circulo), which delimits the reconstruction, this to avoid sharp edges in the  reconstruction
SNRt = 0.2071; %1/2 bit  % Threshold curve for FSC
thickring = 5;         % use thickring for smoothing 
FSC_vertical_range =1:Nw_sino;  % Choose  vertical region for FSC
%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%

% call "close(45)" if you want to plot FSC to clean window (without overlay with previous FSC curve)
[resolution FSC T freq n FSC_stats, fsc_path] = tomo.get_FSC_from_subtomos(tomograms, FSC_vertical_range, rad_apod, radial_smooth, axial_apod,SNRt,thickring,par);
utils.savefast_safe([fsc_path, '.mat'], 'FSC','T','freq','n','FSC_stats', 'rad_apod','axial_apod','theta','total_shift', 'par', par.force_overwrite || par.online_tomo);
clear tomograms 


%% Full tomogram - get a final full resolution reconstruction 
% remove artefacts around edges of tomogram 
if par.usecircle
    [~,circulo] = utils.apply_3D_apodization(ones(Npix), apodize, 0, radial_smooth_apodize); 
end

[Nw_sino,Nl_sino,~] = size(sinogram);

% generate configuration
[cfg, vectors] = astra.ASTRA_initialize([Npix,Npix, Nw_sino],[Nw_sino,Nl_sino],theta,par.lamino_angle,par.tilt_angle); 
% find optimal split of the dataset for given GPU 
split = astra.ASTRA_find_optimal_split(cfg, length(par.GPU_list), 1);

clear tomogram
tomogram = tomo.FBP_zsplit(sinogram, cfg, vectors, split,'valid_angles',par.valid_angles,...
    'GPU', par.GPU_list,'filter',par.filter_type, 'filter_value',par.freq_scale,...
    'use_derivative', strcmpi(par.unwrap_data_method, 'none'), 'mask', circulo);

clear sinogram
% calculate complex refractive index 
tomogram_delta = tomogram*par.factor;
clear tomogram



%% Quick preview of reconstructed volume 

plotting.smart_figure(5)
plotting.imagesc3D(tomogram_delta, 'init_frame', size(tomogram_delta,3)/2)
colormap bone 
axis image off
drawnow 


%% Display reconstructed volume by slices
%%%%%%%%%%%%%%%%%%%%%%%%%
%%% Edit this section %%%
%%%%%%%%%%%%%%%%%%%%%%%%%
par.displayaxis = 3;            % 1 coronal, 2 sagital, 3 axial
par.displayslice = Nl_sino/2;    % Many slices supported e.g [100:200]. If left empty the center slice is shown, 
par.animatedslices = false;     % All slices are shown ignoring those in displayslice
par.average_slices = false;    % If true, it ignores animatedslices
% Display options
par.tomobaraxis = 'auto';       % = 'auto', 'auto_per_frame'  or e.g.  = [-1 1]. For auto it picks max and min from whole tomogram
par.scale = 'edensity';            % = 'phase', 'delta' [index] or 'edensity' [electron/A^3]
par.colormapchoice = 'bone';    % Choose the colormap
par.realaxis = true;            % = true to show result with real axes units, = false for pixels
par.reverse_contrast = true;    % Reverse grayscale
% Scale bar options (only shown if realaxis = true)
par.bar_length= 2e-6;           % Lenght of scalebar (only used if realaxis = true)
par.bar_height= 0.2e-6;           % Height of scalebar (only used if realaxis = true)
par.bar_start_point=[0.8 0.2]*1e-6;    % (x, y) starting point of the scale bar (from corner) in meters
par.bar_color = 'k';            % Color of scalebar
% Other
par.windowautopos = false;      % If true the window position and size is changed, can be useful as false when ready to render a movie
par.makemovie = false;          % = true for rendering an AVI
par.writesnapshots = true;      % Makes jpeg and eps snaphots. It overwrites snapshots. If many slices are chosen it writes only the last one.
par.pausetime = 0.1;            %   use time for animation or multiple slices
%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%

% Coronal, first index
% Sagital, second index
% Axial,   third index

tomo.show_tomogram_cuts(tomogram_delta, par.scanstomo, par)


%% Save Tomogram (delta)
%%%%%%%%%%%%%%%%%%%%%%%%%
%%% Edit this section %%%
%%%%%%%%%%%%%%%%%%%%%%%%%
savedata = true;
%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%
% Note saving tomogram in delta (index of refraction)
% To get phase tomogram   = tomogram_delta/par.factor
% To get electron density tomogram in [e/A^3]   = tomogram_delta*par.factor_edensity
% If you saved before you may clear 
if savedata && ~debug()
    tomo.save_tomogram(tomogram_delta, par, 'delta', circulo, theta)
end



%% Save Tiff files for 3D visualization with external program
%%%%%%%%%%%%%%%%%%%%%%%%%
%%% Edit this section %%%
%%%%%%%%%%%%%%%%%%%%%%%%%
par.save_as_stack = false; 
par.tiff_compression = 'none';
par.tiff_subfolder_name = ['TIFF_delta_' par.filter_type '_freqscl_' sprintf('%0.2f',par.freq_scale)];
par.name_prefix = 'tomo_delta';
%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%
if ~debug()
    tomo.save_as_tiff(tomogram_delta, par)
end



utils.verbose(-1,'=========== INTERIOR TOMO-ALIGNMENT AND RECONSTRUCTION FINISHED ==============')
reset(gpuDevice)  % make sure that the GPU memory is empty 




%*-----------------------------------------------------------------------*
%|                                                                       |
%|  Except where otherwise noted, this work is licensed under a          |
%|  Creative Commons Attribution-NonCommercial-ShareAlike 4.0            |
%|  International (CC BY-NC-SA 4.0) license.                             |
%|                                                                       |
%|  Copyright (c) 2017 by Paul Scherrer Institute (http://www.psi.ch)    |
%|                                                                       |
%|       Author: CXS group, PSI                                          |
%*-----------------------------------------------------------------------*
% You may use this code with the following provisions:
%
% If the code is fully or partially redistributed, or rewritten in another
%   computing language this notice should be included in the redistribution.
%
% If this code, or subfunctions or parts of it, is used for research in a 
%   publication or if it is fully or partially rewritten for another 
%   computing language the authors and institution should be acknowledged 
%   in written form in the publication: “Data processing was carried out 
%   using the “cSAXS matlab package” developed by the CXS group,
%   Paul Scherrer Institut, Switzerland.” 
%   Variations on the latter text can be incorporated upon discussion with 
%   the CXS group if needed to more specifically reflect the use of the package 
%   for the published work.
%
% A publication that focuses on describing features, or parameters, that
%    are already existing in the code should be first discussed with the
%    authors.
%   
% This code and subroutines are part of a continuous development, they 
%    are provided “as they are” without guarantees or liability on part
%    of PSI or the authors. It is the user responsibility to ensure its 
%    proper use and the correctness of the results.
