% tomo_quantitative.m

base_path='../../../';  % when working in utils
tomo_read_path=base_path; % example for offline tomo: '/das/work/units/csaxs/p17649/amplitude_tomo_Ana/'
addpath('utils')
addpath(find_base_package())

return


%% Constants:

scrsz = get(0,'ScreenSize');
tomo_folder='tomo_offline_id_167_SOEC_aged_high_T_S01824_to_S02097__recons_';
q.tomo_path_read= fullfile(base_path, 'analysis_tomo', tomo_folder); 
q.tomo_file_name='tomogram_delta_offline_id_167_SOEC_aged_high_T_S01824_to_S02097_FBP_ram-lak_freqscl_1.00.mat';

%% Read tomographic reconstruction:

load(fullfile(q.tomo_path_read,q.tomo_file_name));
q.tomo_path_write= fullfile(q.tomo_path_read,sprintf('quantitative_%s_%4.2f/',par.filter_type,par.freq_scale)); 
q.pixel_size=par.pixel_size;

%% Make histogram of whole sample

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%Choose parameters
sam=1000;      % Number of bins in histogram
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

% Remove data ouside computed tomogram
N = size(tomogram_delta,1);
xt = [-N/2:N/2-1];
[Xt Yt] = meshgrid(xt,xt);
circulo = 1-utils.radtap(Xt,Yt,10,N/2-3);
cylinder=repmat(circulo,[1 1 size(tomogram_delta,3)]);
data=tomogram_delta.*cylinder;

% Calculate whole histogram
M=size(data,1)*size(data,2)*size(data,3);
data_long=reshape(data,M,1);
cylinder_long=reshape(cylinder,M,1);
data_nozeros=data_long(cylinder_long == 1);
[hst,bins]=hist(data_nozeros,sam);

clear circulo
clear cylinder
clear tomogram_delta

%% Plot histogram

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%Choose parameters
quant='eden';  % Choose quantity to plot: 'delta' for delta or 'eden' for electron density
yaxis='lin';   % Y axis can be linear ('lin') or logaritmic ('log')
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

figure(1);

if isstr(quant)&&strcmpi('eden',quant)
    bins_plot=bins*par.factor_edensity;
    xaxis_label=sprintf('electron density (%c^{-3})',197);
elseif isstr(quant)&&strcmpi('delta',quant)
    bins_plot=bins;
    xaxis_label='delta';
else
    error('Supported strings for quant are delta or eden')
end

if isstr(yaxis)&&strcmpi('lin',yaxis)
    plot(bins_plot,hst); xlabel(xaxis_label); ylabel('number of voxels');
elseif  isstr(yaxis)&&strcmpi('log',yaxis)
    semilogy(bins_plot,hst); xlabel(xaxis_label); ylabel('number of voxels');
else
    error('Supported strings for yaxis are lin or log')
end 

%% Save histogram data

savedata=0;  % Equal to 1 for saving data, or 0 for not saving

if savedata
    if isfolder(q.tomo_path_write) == 0
        mkdir(q.tomo_path_write)
    end
    fid=fopen(sprintf('%shistogram_%s.txt',q.tomo_path_write,tomo_folder),'w');
    fprintf(fid, '# delta \t electron density (Ansgtrom-3) \t  number of voxels\n');
    for hh=1:length(bins)
        fprintf(fid, '%e \t %e \t %e\n', bins(hh),par.factor_edensity*bins(hh),hst(hh));
    end
    fclose(fid);
    
    save(sprintf('%shistogram_%s.mat',q.tomo_path_write,tomo_folder),'bins','hst','q');
   
    print('-f1','-depsc2', [ q.tomo_path_write sprintf('histogram_%s.eps',tomo_folder)]);
    print('-f1','-dpng', [ q.tomo_path_write sprintf('histogram_%s.png',tomo_folder)]);

end
    

%% Plot slices to navigate in 3D data (with color lines)

%Choose parameters %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
q.analysis_case='top';   % please chose a different name for different slected volumes to save data in separate folders
q.quant='eden';             % choose quantity to plot: 'delta' for delta or 'eden' for electron density
scl=[0.8 2];               % color scale can be 'auto' for automatic or e.g. [0.25 0.45]
q.valz=430;                 % z coordinate to select slice in xy plane
q.valx=280;                 % x coordinate to select slice in yz plane
q.valy=280;                 % y coordinate to select slice in xz plane
q.sidex=200;                 % box size in x for volume of interest
q.sidey=200;                 % box size in y for volume of interest
q.sidez=160; % box size in z for volume of interest
% q.valz=470;                 % z coordinate to select slice in xy plane
% q.valx=241;                 % x coordinate to select slice in yz plane
% q.valy=280;                 % y coordinate to select slice in xz plane
% q.sidex=26;                 % box size in x for volume of interest
% q.sidey=26;                 % box size in y for volume of interest
% q.sidez=26; 
colorx='r';
colory='b';
colorz='g';
color_map='jet';
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

xs=q.valx-round(q.sidex/2);
xf=q.valx+round(q.sidex/2);
ys=q.valy-round(q.sidey/2);
yf=q.valy+round(q.sidey/2);
zs=q.valz-round(q.sidez/2);
zf=q.valz+round(q.sidez/2);

if isstr(q.quant)&&strcmpi('eden',q.quant)
    data_corr=data*par.factor_edensity;
    xaxis_label=sprintf('electron density (%c^{-3})',197);
elseif isstr(q.quant)&&strcmpi('delta',q.quant)
    data_corr=data;
    xaxis_label='delta';
else
    error('Supported strings for q.quant are delta or eden')
end

if isstr(scl)&&strcmpi('auto',scl)
    q.scale=[min(data_corr(:)) max(data_corr(:))];
else
    q.scale=scl;
end

figure(2);
%figure('Position',[1,400,800,800]); 
subplot(2,2,3);
imagesc(data_corr(:,:,q.valz), q.scale); axis xy equal tight;
xlabel('x'); ylabel('y')
title(sprintf('z = %d',q.valz)); colormap bone(256); hold on;
plot([q.valx,q.valx],[1,size(data_corr,1)],colorx);
plot([1,size(data_corr,2)],[q.valy,q.valy],colory);
plot([1,size(data_corr,2)],[1,1],colorz,'Linewidth',3);
plot([1,size(data_corr,2)],[size(data_corr,1),size(data_corr,1)],colorz,'Linewidth',3);
plot([1,1],[1,size(data_corr,1)],colorz,'Linewidth',3);
plot([size(data_corr,2),size(data_corr,2)],[1,size(data_corr,1)],colorz,'Linewidth',3);
plot([xs,xf],[ys,ys],colorz);
plot([xs,xf],[yf,yf],colorz);
plot([xs,xs],[ys,yf],colorz);
plot([xf,xf],[ys,yf],colorz);
hold off;

subplot(2,2,4);
imageyz=(squeeze(data_corr(:,q.valx,:)));
imagesc(imageyz, q.scale); axis xy equal tight; colorbar;
h=colorbar;
set(get(h,'label'),'string',xaxis_label); 
xlabel('z'); ylabel('y');
title(sprintf('x = %d',q.valx)); colormap bone(256); hold on;
plot([q.valz,q.valz],[1,size(data_corr,1)],colorz);
plot([1,size(data_corr,3)],[q.valy,q.valy],colory);
plot([1,size(data_corr,3)],[1,1],colorx,'Linewidth',3);
plot([1,size(data_corr,3)],[size(data_corr,1),size(data_corr,1)],colorx,'Linewidth',3);
plot([1,1],[1,size(data_corr,1)],colorx,'Linewidth',3);
plot([size(data_corr,3),size(data_corr,3)],[1,size(data_corr,1)],colorx,'Linewidth',3);
plot([zs,zf],[ys,ys],colorx);
plot([zs,zf],[yf,yf],colorx);
plot([zs,zs],[ys,yf],colorx);
plot([zf,zf],[ys,yf],colorx);
hold off;

subplot(2,2,1);
imagexz=(squeeze(data_corr(q.valy,:,:)))';
imagesc(imagexz, q.scale); axis xy equal tight;
xlabel('x'); ylabel('z');
title(sprintf('y = %d',q.valy)); colormap bone(256); hold on;
plot([q.valx,q.valx],[1,size(data_corr,3)],colorx);
plot([1,size(data_corr,2)],[q.valz,q.valz],colorz);
plot([1,size(data_corr,2)],[1,1],colory,'Linewidth',3);
plot([1,size(data_corr,2)],[size(data_corr,3),size(data_corr,3)],colory,'Linewidth',3);
plot([1,1],[1,size(data_corr,3)],colory,'Linewidth',3);
plot([size(data_corr,2),size(data_corr,2)],[1,size(data_corr,3)],colory,'Linewidth',3);
plot([xs,xf],[zs,zs],colory);
plot([xs,xf],[zf,zf],colory);
plot([xs,xs],[zs,zf],colory);
plot([xf,xf],[zs,zf],colory);
hold off;
set(gcf,'Outerposition',[1 5 800 800])

%% Histogram of selected voi

% Choose parameters %%%%%%%%%%%%%%%%
q.sampling_sel=100;              % Number of bins in histogram
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

data_sel=data_corr(ys:yf,xs:xf,zs:zf);

figure(3);
%figure('Position',[1,400,800,800]); 
subplot(2,2,3);
imagesc(data_sel(:,:,round((zf-zs)/2)), q.scale); axis xy equal tight;
xlabel('x'); ylabel('y')
title(sprintf('z = %d',q.valz)); colormap bone(256); hold on;
hold off;

subplot(2,2,4);
imagesc(squeeze(data_sel(:,round((xf-xs)/2),:)), q.scale); axis xy equal tight;
h=colorbar;
set(get(h,'label'),'string',xaxis_label); 
xlabel('z'); ylabel('y');
title(sprintf('x = %d',q.valx)); colormap bone(256); hold on;
hold off;

subplot(2,2,1);
imagesc(squeeze(data_sel(round((yf-ys)/2),:,:))', q.scale); axis xy equal tight;
xlabel('x'); ylabel('z');
title(sprintf('y = %d',q.valy)); colormap bone(256); hold on;
hold off;

M_sel=size(data_sel,1)*size(data_sel,2)*size(data_sel,3);
data_sel_long=reshape(data_sel,M_sel,1);
[hst_sel,bins_sel]=hist(data_sel_long,q.sampling_sel);
q.hst_sel=hst_sel;
q.bins_sel=bins_sel;

figure(4)
plot(q.bins_sel, q.hst_sel)
xlabel(xaxis_label)
ylabel('number of voxels')
title('histogram of VOI')

%% Make individual plots without lines

x=((1:size(data_corr,2))-round(size(data_corr,2))/2)*q.pixel_size*1e6; % [microns]
y=((1:size(data_corr,1))-round(size(data_corr,1))/2)*q.pixel_size*1e6; % [microns]
z=((1:size(data_corr,3))-round(size(data_corr,3))/2)*q.pixel_size*1e6; % [microns]

figure(5)
imagexz=(squeeze(data_corr(q.valy,:,:)))';
imagesc(x,z,imagexz, q.scale); axis xy equal tight;
xlabel('x (microns)'); ylabel('z (microns)');
title(sprintf('%s; y = %d',xaxis_label,q.valy)); colormap bone(256); 
h=colorbar;colormap('bone');
set(get(h,'label'),'string',xaxis_label); 

figure(6)
imageyz=(squeeze(data_corr(:,q.valx,:)))';
imagesc(y,z,imageyz, q.scale); axis xy equal tight; colorbar;
xlabel('y (microns)'); ylabel('z (microns)');
title(sprintf('%s; x = %d',xaxis_label,q.valx)); colormap bone(256); 
h=colorbar;colormap('bone');
set(get(h,'label'),'string',xaxis_label); 

figure(7)
imagesc(x,y,data_corr(:,:,q.valz), q.scale); axis xy equal tight;
title(sprintf('%s; z = %d',xaxis_label,q.valz)); colormap bone(256); 
xlabel('x (microns)'); ylabel('y (microns)');
h=colorbar;colormap('bone');
set(get(h,'label'),'string',xaxis_label); 

%% Fit histogram peak to Gaussian curve

q.fit_type='gauss1'; % try 'gauss1' for one peak and 'gauss2' for a double peak fit
weight=ones(size(q.hst_sel)); %Example: (q.bins_sel > 0.5).*(q.bins_sel < 1.5);
q.f = fit(q.bins_sel.',q.hst_sel.',q.fit_type,'Weights',weight);  
q.f
figure(8)
plot(q.f,q.bins_sel,q.hst_sel)

q.value=q.f.b1;
q.sigma=q.f.c1/sqrt(2);
q.FWHM=2.35482*q.sigma;

if isstr(q.quant)&&strcmpi('eden',q.quant)
    display(sprintf('electron density: %f4.2 +/- %f4.2',q.value,q.sigma))
else isstr(q.quant)&&strcmpi('delta',q.quant)
    display(sprintf('delta: %e +/- %e',q.value*par.factor_edensity,q.sigma*par.factor_edensity))
end

if isstr(q.fit_type)&&strcmpi('gauss2',q.fit_type)
    q.value2=q.f.b2;
    q.sigma2=q.f.c2/sqrt(2);
    q.FWHM2=2.35482*q.sigma2;
    if isstr(q.quant)&&strcmpi('eden',q.quant)
        display(sprintf('electron density: %f4.2 +/- %f4.2',q.value2,q.sigma2))
    else isstr(q.quant)&&strcmpi('eden',q.quant)
        display(sprintf('delta: %e +/- %e',q.value2*par.factor_edensity,q.sigma2*par.factor_edensity))
    end
    
end
    
%% Estimate mass density
% Choose parameters %%%%%%%%%%%%%%%%%%%
q.AZ_ratio=2.0; % Estimation of molar mass (g/mol)
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

NA=6.022e23;  %[mol-1]

if isstr(q.quant)&&strcmpi('eden',q.quant)
   q.mass_density=q.value*q.AZ_ratio/NA*1e24;
   q.mass_density_sigma=q.sigma*q.AZ_ratio/NA*1e24;
else isstr(q.quant)&&strcmpi('delta',q.quant)
    q.mass_density=q.value*par.factor_edensity*q.AZ_ratio/NA*1e24;
    q.mass_density_sigma=q.sigma*par.factor_edensity*q.AZ_ratio/NA*1e24;
end

if isstr(q.fit_type)&&strcmpi('gauss2',q.fit_type)
  if isstr(q.quant)&&strcmpi('eden',q.quant)
   q.mass_density2=q.value2*q.AZ_ratio/NA*1e24;
   q.mass_density_sigma2=q.sigma2*q.AZ_ratio/NA*1e24;
else isstr(q.quant)&&strcmpi('delta',q.quant)
    q.mass_density2=q.value2*par.factor_edensity*q.AZ_ratio/NA*1e24;
    q.mass_density_sigma2=q.sigma2*factor_edensity*q.AZ_ratio/NA*1e24;
  end
end

display(sprintf('mass density: %f4.2 +/- %f4.2',q.mass_density,q.mass_density_sigma))

if isstr(q.fit_type)&&strcmpi('gauss2',q.fit_type)
   display(sprintf('mass density: %f4.2 +/- %f4.2',q.mass_density2,q.mass_density_sigma2))
end
%% Save analysis
savedata=1;
casefolder=[q.tomo_path_write q.analysis_case '/'];
savename=['histogram_VOI_' q.analysis_case];
if savedata == 1
    if ~exist('casefolder','dir'); mkdir(casefolder); end
    print('-f2','-depsc2', [ casefolder savename '_3D_orientation_wholw_volume.eps']);
    print('-f2','-dtiff', [ casefolder savename '_3D_orientation_whole_volume.tif']);
    print('-f3','-depsc2', [ casefolder savename '_3D_orientation.eps']);
    print('-f3','-dtiff', [ casefolder savename '_3D_orientation.tif']);
    print('-f4','-depsc2', [ casefolder savename '_histogram.eps']);
    print('-f4','-dtiff', [ casefolder savename '_histogram.tif']);
    print('-f5','-depsc2', [ casefolder savename '_slice_y.eps']);
    print('-f5','-dtiff', [ casefolder savename '_slice_y.tif']);
    print('-f6','-depsc2', [ casefolder savename '_slice_x.eps']);
    print('-f6','-dtiff', [ casefolder savename '_slice_x.tif']);
    print('-f7','-depsc2', [ casefolder savename '_slice_z.eps']);
    print('-f7','-dtiff', [ casefolder savename '_slice_z.tif']);
    print('-f8','-depsc2', [ casefolder savename '_Gauss_fit.eps']);
    print('-f8','-dtiff', [ casefolder savename '_Gauss_fit.tif']);

    fid=fopen([casefolder savename 'histogram.txt'],'w');
    if isstr(q.quant)&&strcmpi('eden',q.quant)
        fprintf(fid, '# electron density (Angtrom-3) /  number of voxels\n');
    elseif isstr(q.quant)&&strcmpi('delta',q.quant)
        fprintf(fid, '# delta /  number of voxels\n');
    else
        error('Supported strings for q.quant are delta or eden')
    end
    for hh=1:length(q.bins_sel)
        fprintf(fid, '%e %e\n', q.bins_sel(hh),q.hst_sel(hh));
    end
    fclose(fid);
    
    save([casefolder savename '.m'],'q');
      
end

   
%% Delete large variables
% After this the code needs to be run from the very beginning to read the
% full tomogram
clear data0
clear data_corr

%% Read amplitude data:

qa.tomo_path_read=q.tomo_path_read;
qa.tomo_file_name_amp='tomogram_beta_offline_id_167_SOEC_aged_high_T_S01824_to_S02097_FBP_ram-lak_freqscl_0.25.mat';
ampdata=load(fullfile(qa.tomo_path_read, qa.tomo_file_name_amp)); % tomorec
qa.tomo_path_write= fullfile(qa.tomo_path_read,sprintf('quantitative_beta_%s_%4.2f/',ampdata.par.filter_type_amp,ampdata.par.freq_scale_amp)); 
data_amp_sel=ampdata.tomogram_beta(ys:yf,xs:xf,zs:zf);

qa.pixel_size=ampdata.par.pixel_size;
qa.valx=q.valx;
qa.valy=q.valy;
qa.valz=q.valz;
qa.sidex=q.sidex;
qa.sidey=q.sidey;
qa.sidez=q.sidez;
qa.analysis_case=q.analysis_case;

%% Make beta histogram of whole sample

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%Choose parameters
sam=1000;      % Number of bins in histogram
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

% Remove data ouside computed tomogram
N = size(ampdata.tomogram_beta,1);
xt = [-N/2:N/2-1];
[Xt Yt] = meshgrid(xt,xt);
circulo = 1-utils.radtap(Xt,Yt,10,N/2-3);
cylinder=repmat(circulo,[1 1 size(ampdata.tomogram_beta,3)]);
data=ampdata.tomogram_beta.*cylinder;

% Calculate whole histogram
M=size(data,1)*size(data,2)*size(data,3);
data_long=reshape(data,M,1);
cylinder_long=reshape(cylinder,M,1);
data_nozeros=data_long(cylinder_long == 1);
[hst_amp,bins_amp]=hist(data_nozeros,sam);

clear circulo
clear cylinder

%% Plot histogram

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%Choose parameters
yaxis='lin';   % Y axis can be linear ('lin') or logaritmic ('log')
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

figure(100);
xaxis_label_amp='beta';

if isstr(yaxis)&&strcmpi('lin',yaxis)
    plot(bins_amp,hst_amp); xlabel(xaxis_label_amp); ylabel('number of voxels');
elseif  isstr(yaxis)&&strcmpi('log',yaxis)
    semilogy(bins_amp,hst_amp); xlabel(xaxis_label_amp); ylabel('number of voxels');
else
    error('Supported strings for yaxis are lin or log')
end 

%% Save histogram data

savedata=0;  % Equal to 1 for saving data, or 0 for not saving

if savedata
    if isfolder(qa.tomo_path_write) == 0
        mkdir(qa.tomo_path_write)
    end
    fid=fopen(sprintf('%shistogram_%s.txt',qa.tomo_path_write,tomo_folder),'w');
    fprintf(fid, '# beta \t  number of voxels\n');
    for hh=1:length(bins_amp)
        fprintf(fid, '%e \t %e\n', bins_amp(hh),hst_amp(hh));
    end
    fclose(fid);
    
    save(sprintf('%shistogram_%s.mat',qa.tomo_path_write,tomo_folder),'bins_amp','hst_amp','qa');
   
    print('-f100','-depsc2', [ qa.tomo_path_write sprintf('histogram_%s.eps',tomo_folder)]);
    print('-f100','-dpng', [ qa.tomo_path_write sprintf('histogram_%s.png',tomo_folder)]);

end

%% Plot full amplitude slices to navigate in 3D data (with color lines)
scl_amp=[-0.2e-6,3e-6];  % choose gray scale for plotting beta, e.g. [0,2e-6] or 'auto' for automatic

if isstr(scl)&&strcmpi('auto',scl)
    qa.scale_amp=[min(data_amp_sel(:)) max(data_amp_sel(:))];
else
    qa.scale_amp=scl_amp;
end

figure(11);
%figure('Position',[1,400,800,800]); 
subplot(2,2,3);
imagesc(ampdata.tomogram_beta(:,:,qa.valz), qa.scale_amp); axis xy equal tight;
xlabel('x'); ylabel('y')
title(sprintf('z = %d',qa.valz)); colormap bone(256); hold on;
plot([qa.valx,qa.valx],[1,size(ampdata.tomogram_beta,1)],colorx);
plot([1,size(ampdata.tomogram_beta,2)],[qa.valy,qa.valy],colory);
plot([1,size(ampdata.tomogram_beta,2)],[1,1],colorz,'Linewidth',3);
plot([1,size(ampdata.tomogram_beta,2)],[size(ampdata.tomogram_beta,1),size(ampdata.tomogram_beta,1)],colorz,'Linewidth',3);
plot([1,1],[1,size(ampdata.tomogram_beta,1)],colorz,'Linewidth',3);
plot([size(ampdata.tomogram_beta,2),size(ampdata.tomogram_beta,2)],[1,size(ampdata.tomogram_beta,1)],colorz,'Linewidth',3);
plot([xs,xf],[ys,ys],colorz);
plot([xs,xf],[yf,yf],colorz);
plot([xs,xs],[ys,yf],colorz);
plot([xf,xf],[ys,yf],colorz);
hold off;

subplot(2,2,4);
imageyz=(squeeze(ampdata.tomogram_beta(:,qa.valx,:)));
imagesc(imageyz, qa.scale_amp); axis xy equal tight; colorbar;
xlabel('z'); ylabel('y');
title(sprintf('x = %d',qa.valx)); colormap bone(256); hold on;
plot([qa.valz,qa.valz],[1,size(ampdata.tomogram_beta,1)],colorz);
plot([1,size(ampdata.tomogram_beta,3)],[qa.valy,qa.valy],colory);
plot([1,size(ampdata.tomogram_beta,3)],[1,1],colorx,'Linewidth',3);
plot([1,size(ampdata.tomogram_beta,3)],[size(ampdata.tomogram_beta,1),size(ampdata.tomogram_beta,1)],colorx,'Linewidth',3);
plot([1,1],[1,size(ampdata.tomogram_beta,1)],colorx,'Linewidth',3);
plot([size(ampdata.tomogram_beta,3),size(ampdata.tomogram_beta,3)],[1,size(ampdata.tomogram_beta,1)],colorx,'Linewidth',3);
plot([zs,zf],[ys,ys],colorx);
plot([zs,zf],[yf,yf],colorx);
plot([zs,zs],[ys,yf],colorx);
plot([zf,zf],[ys,yf],colorx);
hold off;

subplot(2,2,1);
imagexz=(squeeze(ampdata.tomogram_beta(qa.valy,:,:)))';
imagesc(imagexz, qa.scale_amp); axis xy equal tight;
xlabel('x'); ylabel('z');
title(sprintf('y = %d',qa.valy)); colormap bone(256); hold on;
plot([qa.valx,qa.valx],[1,size(ampdata.tomogram_beta,3)],colorx);
plot([1,size(ampdata.tomogram_beta,2)],[qa.valz,qa.valz],colorz);
plot([1,size(ampdata.tomogram_beta,2)],[1,1],colory,'Linewidth',3);
plot([1,size(ampdata.tomogram_beta,2)],[size(ampdata.tomogram_beta,3),size(ampdata.tomogram_beta,3)],colory,'Linewidth',3);
plot([1,1],[1,size(ampdata.tomogram_beta,3)],colory,'Linewidth',3);
plot([size(ampdata.tomogram_beta,2),size(ampdata.tomogram_beta,2)],[1,size(ampdata.tomogram_beta,3)],colory,'Linewidth',3);
plot([xs,xf],[zs,zs],colory);
plot([xs,xf],[zf,zf],colory);
plot([xs,xs],[zs,zf],colory);
plot([xf,xf],[zs,zf],colory);
hold off;
set(gcf,'Outerposition',[1 300 800 800])
%% Make individual plots without lines

figure(12)
imagexz=(squeeze(ampdata.tomogram_beta(qa.valy,:,:)))';
imagesc(x,z,imagexz, qa.scale_amp); axis xy equal tight;
xlabel('x (microns)'); ylabel('z (microns)');
title(sprintf('%s; y = %d',xaxis_label_amp,qa.valy)); colormap bone(256); 
colorbar;

figure(13)
imageyz=(squeeze(ampdata.tomogram_beta(:,qa.valx,:)))';
imagesc(y,z,imageyz, qa.scale_amp); axis xy equal tight; colorbar;
xlabel('y (microns)'); ylabel('z (microns)');
title(sprintf('%s; x = %d',xaxis_label_amp,qa.valx)); colormap bone(256); 
colorbar;

figure(14)
imagesc(x,y,ampdata.tomogram_beta(:,:,qa.valz), qa.scale_amp); axis xy equal tight;
title(sprintf('%s; z = %d',xaxis_label_amp,qa.valz)); colormap bone(256); 
xlabel('x (microns)'); ylabel('y (microns)');
colorbar

%% Histogram of selected amplitude voi
qa.sampling_amp_sel=70;

figure(15);
%figure('Position',[1,400,800,800]); 
subplot(2,2,3);
imagesc(data_amp_sel(:,:,round((zf-zs)/2)), qa.scale_amp); axis xy equal tight;
xlabel('x'); ylabel('y')
title(sprintf('z = %d',qa.valz)); colormap bone(256); hold on;
hold off;

subplot(2,2,4);
imagesc(squeeze(data_amp_sel(:,round((xf-xs)/2),:)), qa.scale_amp)
xlabel('z'); ylabel('y');
title(sprintf('x = %d',qa.valx)); colormap bone(256); hold on;
hold off;

subplot(2,2,1);
imagesc(squeeze(data_amp_sel(round((yf-ys)/2),:,:))', qa.scale_amp)
xlabel('x'); ylabel('z');
title(sprintf('y = %d',qa.valy)); colormap bone(256); hold on;
hold off;

M_amp_sel=size(data_amp_sel,1)*size(data_amp_sel,2)*size(data_amp_sel,3);
data_amp_sel_long=reshape(data_amp_sel,M_amp_sel,1);
[hst_amp_sel,bins_amp_sel]=hist(data_amp_sel_long,qa.sampling_amp_sel);

qa.hst_amp_sel=hst_amp_sel;
qa.bins_amp_sel=bins_amp_sel;

figure(16)
plot(qa.bins_amp_sel, qa.hst_amp_sel)
xlabel(xaxis_label_amp)
ylabel('number of voxels')
title('histogram of VOI')

%% Fit histogram peak to Gaussian curve

qa.fit_type='gauss2'; % try 'gauss1' for one peak and 'gauss2' for a double peak fit

weights=ones(size(qa.hst_amp_sel)); % Example: (qa.bins_amp_sel > -2e-6) .* (qa.bins_amp_sel < 4e-6);
qa.f = fit(qa.bins_amp_sel.',qa.hst_amp_sel.',qa.fit_type,'StartPoint',[0.5e5,-0.5e-6,1e-6,2.4e5,1.5e-6,1e-6]);  
qa.f
figure(17)
plot(qa.f,qa.bins_amp_sel,qa.hst_amp_sel)

qa.value=qa.f.b1;
qa.sigma=qa.f.c1/sqrt(2);
qa.FWHM=2.35482*qa.sigma;

display(sprintf('beta: %e +/- %e',qa.value,qa.sigma))

if isstr(qa.fit_type)&&strcmpi('gauss2',qa.fit_type)
    qa.value2=qa.f.b2;
    qa.sigma2=qa.f.c2/sqrt(2);
    qa.FWHM2=2.35482*qa.sigma2;
    display(sprintf('beta: %e +/- %e',qa.value2,qa.sigma2)) 
end
    
%% Save plots with amplitude
savedata=1;
casefolder=[qa.tomo_path_write qa.analysis_case '/'];
savename=['histogram_VOI_beta_' qa.analysis_case];

if savedata == 1
    if ~exist('casefolder','dir'); mkdir(casefolder); end
    print('-f11','-depsc2', [ casefolder savename '_3D_orientation_whole_volume.eps']);
    print('-f11','-dtiff', [ casefolder savename '_3D_orientation_whole_volume.tif']);
    print('-f15','-depsc2', [ casefolder savename '_3D_orientation.eps']);
    print('-f15','-dtiff', [ casefolder savename '_3D_orientation.tif']);
    print('-f16','-depsc2', [ casefolder savename '_histogram.eps']);
    print('-f16','-dtiff', [ casefolder savename '_histogram.tif']);
    print('-f17','-depsc2', [ casefolder savename '_histogram_Gauss_fit.eps']);
    print('-f17','-dtiff', [ casefolder savename '_histogram_Gauss_fit.tif']);
    print('-f12','-depsc2', [ casefolder savename '_slice_y.eps']);
    print('-f12','-dtiff', [ casefolder savename '_slice_y.tif']);
    print('-f13','-depsc2', [ casefolder savename '_slice_x.eps']);
    print('-f13','-dtiff', [ casefolder savename '_slice_x.tif']);
    print('-f14','-depsc2', [ casefolder savename '_slice_z.eps']);
    print('-f14','-dtiff', [ casefolder savename '_slice_z.tif']);
    
    fid=fopen([casefolder savename 'histogram.txt'],'w');
    fprintf(fid, '# beta /  number of voxels\n');
    for hh=1:length(qa.bins_amp_sel)
        fprintf(fid, '%e %e\n', qa.bins_amp_sel(hh),qa.hst_amp_sel(hh));
    end
    fclose(fid);
    
    save([casefolder savename '.m'],'qa');
end
%% Make bivariate histogram of voi

bins = 250; % number of bins of the histogram
spacing = 'lin'; %'lin'; 'log'; % lin is better
delta_slice=data_sel;
abs_slice=data_amp_sel;

% find indices corresponding to the materials phase only (exclude air)
% clear mask mask_ind
mask=data_sel;%>1E-6;
mask_ind=find(delta_slice);%>1E-6);

% Reshape the images into 1D vectors
x=abs_slice(mask_ind);
y=delta_slice(mask_ind);
%x=abs_slice;
%y=delta_slice;

clear xedges yedges
switch lower(spacing)
    case 'lin'
        % linearly spaced edges of the histogram
        xedges = linspace(min(x),max(x),bins);
        yedges = linspace(min(y),max(y),bins);
    case 'log'
        xedges = linspace(min(x),max(x),bins);
        yedges = logspace(log10(min(y)),log10(max(y)),bins);
end

% Calculate the 1D histogram
[xn, xbin] = histc(x,xedges);
[yn, ybin] = histc(y,yedges);

%xbin, ybin zero for out of range values 
% (see the help of histc) force this event to the 
% first bins
xbin(find(xbin == 0)) = inf;
ybin(find(ybin == 0)) = inf;

xnbin = length(xedges);
ynbin = length(yedges);

if xnbin >= ynbin
    xy = ybin*(xnbin) + xbin;
      indexshift =  xnbin; 
else
    xy = xbin*(ynbin) + ybin;
      indexshift =  ynbin; 
end

%[xyuni, m, n] = unique(xy);
xyuni = unique(xy);
xyuni(end) = []; 
hstres = histc(xy,xyuni);
clear xy;

histmat = zeros(ynbin,xnbin);
histmat(xyuni-indexshift) = hstres;

%% display the bivariate histogram

%limits for plotting:
x_range= [-1,3]*1e-6; % amplitude
y_range=[-0.1,2.5];   % delta or electron density


figure(18)
sub1=subplot(3,3,[4,5,7,8]);
imagesc(xedges, yedges, log10(histmat')), axis xy square tight
xlim(x_range);
ylim(y_range);

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%% For drawing the lines %%%%
%lineh1= 1.0; % value in beta .*1e-5
%lineh2= 1.0; % value in beta .*1e-5
%linev1= 2.5; % value in delta .*1e-7
%linev2= 2.5; % value in delta .*1e-7
%%%%%%%   end of edit %%%%%%%%%%%%%%%%
hold on
%plot([-4 12],[lineh1 lineh1],'-b')
%plot([-4 12],[lineh2 lineh2],'-b')
%plot([linev1 linev1],[0.2 2],'-b')
%plot([linev2 linev2],[0.2 2],'-b')
hold off

thisfontsize=12;

colormap('plotting.franzmap')
Contours =[1e0 1e1 1e2 1e3 1e4 1e5 1e6 1e7];
hColorbar = colorbar('East','YTick',log10(Contours),'YTickLabel',Contours);
hXLabel = xlabel(xaxis_label_amp);
hYLabel = ylabel(xaxis_label);
set(gca,...
    'FontName'  , 'Helvetica',...
    'FontSize'  , thisfontsize         ,...
    'Box'       , 'off'      ,...
    'OuterPosition', [0 0 0.53 0.73] ,...
    'TickDir'   , 'out'      ,...'YAxisLocation','right'
    'XMinorTick', 'on'       ,...
    'YMinorTick', 'on'       ,...
    'XColor'    , [.0 .0 .0] ,...   
    'YColor'    , [.0 .0 .0] ,...
    'LineWidth' , 1         );
yticks([0,1]);
set([hXLabel,hYLabel],...
    'FontName', 'Arial',...
    'FontSize', thisfontsize-1 );
pos1=get(sub1,'Position');
set(hColorbar,...
    'Box'    , 'on'         ,...
    'TickDir', 'in'         ,...
    'Direction','normal', ...
    'YAxisLocation','left',...
    'YColor' , [0.9 0.9 0.9] ,...
    'XColor' , [0.9 0.9 0.9] , ...    
    'Position',[0.47 0.16 0.03 0.3]);

sub2=subplot(3,3,[1,2])
b=bar(xedges,xn,1)
b.FaceColor='b';
b.EdgeColor='b';
axis xy tight
xlim(x_range); 
%ylim([0 4])
hYLabel1 = ylabel('Freq.');
set(gca,...
    'FontName'  , 'Helvetica',...
    'FontSize'  , thisfontsize         ,...
    'Box'       , 'off'      ,...
    'OuterPosition', [0.012 0.72 0.53 0.22], ...
    'TickDir'   , 'out'      ,...
    'XMinorTick', 'off'      ,...
    'XTick'     , []         ,...
    'XTickLabel', []         ,...
    'Layer'     , 'top'      ,...
    'YMinorTick', 'on'       ,...
    'XColor'    , [.0 .0 .0] ,...   
    'YColor'    , [.0 .0 .0] ,...
    'LineWidth' , 1         );
set(hYLabel1,...
    'FontName', 'Arial',...
    'FontSize', thisfontsize -1   );
pos2=get(sub2,'Position');
sub3=subplot(3,3,[6,9])
b=barh(yedges,yn,1), 
b.FaceColor='r';
b.EdgeColor='r';
axis xy tight
ylim(y_range);
%xlim([0 20]);
hXLabel1=xlabel('Freq.');
set(gca,...
    'FontName'  , 'Helvetica',...
    'FontSize'  , thisfontsize         ,...
    'Box'       , 'off'      ,...
    'OuterPosition', [0.534 0.0010 0.17 0.796],...
    'TickDir'   , 'out'      ,...
    'XAxisLocation', 'top'   ,...
    'XMinorTick', 'off'      ,...
    'YTick'     , []         ,...
    'YTickLabel', []         ,...
    'Layer'     , 'top'      ,...
    'XMinorTick', 'on'       ,...
    'XColor'    , [.0 .0 .0] ,...   
    'YColor'    , [.0 .0 .0] ,...
    'LineWidth' , 1         );
set(hXLabel1,...
    'FontName', 'Arial',...
    'FontSize', thisfontsize  -1  );
% %xlim([0 10]);
%hXLabel = xlabel('Absorption index, \beta [x 10^{-7}]');
%hYLabel = ylabel('Refractive index decrement, \delta [x 10^{-5}] ');
pos3=get(sub3,'Position');
pos2(3)=pos1(3);
pos3(4)=pos1(4);
set(sub2,'Position',pos2);
set(sub3,'Position',pos3);
%set(figure(18),'OuterPosition',[402 189 800 800])

%% Save bivariate histogram
savedata=1;
qb.tomo_path_write= fullfile(q.tomo_path_read, sprintf('quantitative_%s_%4.2f_beta_%s_%4.2f/',par.filter_type,par.freq_scale,ampdata.par.filter_type_amp,ampdata.par.freq_scale_amp));
qb.analysis_case=qa.analysis_case;
casefolder=fullfile(qb.tomo_path_write, qb.analysis_case);
savename=['histogram_VOI_bivariate' qb.analysis_case];

if savedata == 1
    if ~exist('casefolder','dir'); mkdir(casefolder); end
    print('-f18','-depsc2', [ casefolder savename '.eps']);
    print('-f18','-dtiff', [ casefolder savename '.tif']);    
    
    save([casefolder savename '.m'],'qb');
end
