function vol_full = Atx_partial(projData, cfg, vectors, split)
% recompile commands
%  (Linux)   mexcuda -largeArrayDims -v    ASTRA_GPU_wrapper/ASTRA_GPU_wrapper.cu ASTRA_GPU_wrapper/util3d.cu ASTRA_GPU_wrapper/par3d_fp.cu ASTRA_GPU_wrapper/par3d_bp.cu
%  (Windows)  mexcuda -largeArrayDims -v    ASTRA_GPU_wrapper\ASTRA_GPU_wrapper.cu ASTRA_GPU_wrapper\util3d.cu ASTRA_GPU_wrapper\par3d_fp.cu ASTRA_GPU_wrapper\par3d_bp.cu


    %% backprojector that allows to split the full volume into smaller pieces
    % projData - array Nlayers x width_sinogram x Nangles of back-projected data 
    % cfg - config structure generated by ASTRA_initialize
    % vectors - orientation of projections generated by ASTRA_initialize
    % split  3 or 4 elements vector, [split X, split Y, split Z, split angle ]


    keep_on_GPU =  isa(projData, 'gpuArray');

    assert(max(cfg.iProjU, cfg.iProjV) <= 4096, 'Sinogram exceed max size allowed by GPU')

    split_projections = cfg.iProjAngles > 1024 || ... 
       numel(projData)*4 > 1024e6 || ... ;  % Data array is to large, try splitting
       length(split) > 3;  % split contains also angular split 
  
    projData = matlab2astra(projData);
    
    if all(split == 1) &&  ~split_projections
%         assert(cfg.iProjAngles*cfg.iProjU*cfg.iProjV*4/1e6 / prod(split) < 1024, 'Data array is to large, try splitting')
        vol_full = ASTRA_GPU_wrapper('bp', gpuArray(single(projData)), cfg, vectors);
        if ~keep_on_GPU; vol_full = gather(vol_full); end 
        return
    end

%     assert( all(mod(split,2)==1), 'Split should be odd number')

    if isscalar(split)
        split = split .* ones(3,1);
    end
    
    %% preallocate large array for results 
    Npix_full = [cfg.iVolX,cfg.iVolY,cfg.iVolZ];
    Npix_small = Npix_full./split(1:3);
    
%     fprintf('Size of the full volume: %i %i %i\n', Npix_full)
%     fprintf('Size of the subvolume: %i %i %i\n', Npix_small)
    
    assert( all(mod(Npix_small,1)==0), sprintf('Volume array cannot be divided to %i %i %i cubes', split))
    assert(all(size(projData)==[cfg.iProjU,cfg.iProjV,cfg.iProjAngles]), 'Wrong inputs size')
    assert(all(size(vectors)==[cfg.iProjAngles,12]), 'Wrong vectors size')

    
    gpu  = gpuDevice;
    fprintf('Free GPU memory: %3.2g%%\n',  gpu.AvailableMemory/gpu.TotalMemory*100)

    fprintf('Size of the data: %i %i %i\n', size(projData))

    
    

    cfg.iVolX = Npix_small(1);
    cfg.iVolY = Npix_small(2);
    cfg.iVolZ = Npix_small(3);
    
    if split_projections
        %% fix if the number of projections is > 1024 (or it can be fixed in the ASTRA code )
        projData =  astra2matlab(projData);
        %% if the dataset is too large, do automatic splitting along angles 
        Nproj_groups  = ceil(max(cfg.iProjAngles/1024,  numel(projData)*4 / 1024e6));
        if length(split) > 3 
            Nproj_groups = max(split(4), Nproj_groups);
        end
        
        for i = 1:Nproj_groups
            ind = (1+(i-1)*ceil(cfg.iProjAngles/Nproj_groups)):i*ceil(cfg.iProjAngles/Nproj_groups);
            ind = ind(ind <= cfg.iProjAngles);  
            projData_tmp{i} =  matlab2astra(projData(:,:,ind));
            vectors_tmp{i} = vectors(ind,:);
            cfg_tmp{i} = cfg;
            cfg_tmp{i}.iProjAngles = length(ind);
        end
        projData = projData_tmp;
        cfg = cfg_tmp; vectors = vectors_tmp;
        clear projData_tmp 
    else
        Nproj_groups = 1;
        projData = {projData};
        cfg = {cfg};
        vectors = {vectors};
    end
    
       
    if keep_on_GPU
        % transfer to GPU now 
        vol_full = gpuArray.zeros(Npix_full, 'single');
    else
        %% keep the volume in RAM 
        vol_full = zeros(Npix_full, 'single');
    end
    

    
    iter = 0;
    for k = 1:Nproj_groups
        projData{k} = gpuArray(single(projData{k}));  % transfer projections to GPU if not there yet 
        for z = 1:split(3)
            for x =  1:split(1)
                for y =1:split(2)
                    assert(numel(projData{k}) * 4 < 1024e6, 'Data exceeded maximal size of texture 1024MB')
                    pos = [x,y,z];
                    for n = 1:3
                        %% find optimal shift of the subvolume
                        if mod(split(n),2)==1  %% odd 
                            shift(n) = (pos(n) - ceil(split(n)/2))*Npix_small(n);
                        else
                            shift(n) = (pos(n) - ceil(split(n)/2)-1/2)*Npix_small(n);
                        end
                    end
                     
                    vectors_tmp = vectors{k};
                    vectors_tmp(:,4:6) = bsxfun(@minus, vectors_tmp(:,4:6), shift);

                    vol = ASTRA_GPU_wrapper('bp', projData{k}, cfg{k}, vectors_tmp);
%                     keyboard
                    if ~keep_on_GPU
                        vol = gather(vol);  % return to RAM
                    end
                    % return results to the large array 
                    vol_full=add_to_3D(vol_full, vol,([x,y,z]-1).*Npix_small);

                    iter = iter+1;
                    progressbar(iter, prod(split)*Nproj_groups, 20);

                end
            end
        end
        projData{k} = []; % delete from GPU 
    end
    
end

