% SART_ZSPLIT simple wrapper of the SART function that allows easy dataset
%           splitting along the vertical axis 
% 
%   rec=SART_zsplit(rec, sinogram, cfg, vectors, cache, relax )
%
% Inputs:
%     **rec         - initial guess of the reconstruction 
%     **sino        - sinogram (Nlayers x width x Nangles)
%     **cfg         - config struct from ASTRA_initialize
%     **vectors     - vectors of projection rotation generated by ASTRA_initialize
%     **Niter       - number of iterations
%     **relax       - 0 == no relaxation, -> relax projection in presence of noise 
%     **split       - split parameter for the ASTRA projector 
%  *optional*
%     ** split =[1,1,1]           - split the solved volume, split(3 is used to split in separated blocks, split(1:2) is used inside Atx_partial to du subplitting for ASTRA 
%     ** valid_angles =  []       - list of valid angles, []==all are valid
%     ** filter =   'ram-lak'     - name of the FBP filter 
%     ** filter_value =   1       - fitlering value for the FBP filter 
%     ** deformation_fields = {}  - cell 3x1 of deformation arrays 
%     ** inv_deformation_fields={}-cell 3x1 of inverse deformation arrays 
%     ** GPU =  []                - list of GPUs to be used in reconstruction
%     ** split_sub =  [1,1,1]     - splitting of the sub block on smaller tasks in the Atx_partial method , 1 == no splitting 
%     ** verbose =  1             - verbose = 0 : quiet, verbose : standard info , verbose = 2: debug 
%     ** relax = 0                - relaxation constant, shorter steps , 1 == no relaxation 
%     ** constraint= []           - constraint function that will be applied after every step 
%
% *returns*
%    ++rec          - FBP reconstruction 

%*-----------------------------------------------------------------------*
%|                                                                       |
%|  Except where otherwise noted, this work is licensed under a          |
%|  Creative Commons Attribution-NonCommercial-ShareAlike 4.0            |
%|  International (CC BY-NC-SA 4.0) license.                             |
%|                                                                       |
%|  Copyright (c) 2017 by Paul Scherrer Institute (http://www.psi.ch)    |
%|                                                                       |
%|       Author: CXS group, PSI                                          |
%*-----------------------------------------------------------------------*
% You may use this code with the following provisions:
%
% If the code is fully or partially redistributed, or rewritten in another
%   computing language this notice should be included in the redistribution.
%
% If this code, or subfunctions or parts of it, is used for research in a 
%   publication or if it is fully or partially rewritten for another 
%   computing language the authors and institution should be acknowledged 
%   in written form in the publication: “Data processing was carried out 
%   using the “cSAXS matlab package” developed by the CXS group,
%   Paul Scherrer Institut, Switzerland.” 
%   Variations on the latter text can be incorporated upon discussion with 
%   the CXS group if needed to more specifically reflect the use of the package 
%   for the published work.
%
% A publication that focuses on describing features, or parameters, that
%    are already existing in the code should be first discussed with the
%    authors.
%   
% This code and subroutines are part of a continuous development, they 
%    are provided “as they are” without guarantees or liability on part
%    of PSI or the authors. It is the user responsibility to ensure its 
%    proper use and the correctness of the results.



function rec=SART_zsplit(rec, sinogram, cfg, vectors, cache, relax )


    [Nlayers, ~, ~] = size(sinogram); 
    Nvols = ceil(numel(rec)*4 / 1024e6);
    Nlayers_on_GPU = ceil(Nlayers/Nvols); 

    for i = 1:ceil(Nlayers/Nlayers_on_GPU)       
        progressbar(i, ceil(Nlayers/Nlayers_on_GPU))
        ind = (1+(i-1)*Nlayers_on_GPU):min(i*Nlayers_on_GPU, Nlayers);
        %%%%%%%%%%%%%%%%%%%% SART SPLIT + MOVE TO GPU %%%%%%%%%%%%%%%%%%%%%%%
        rec_tmp = gpuArray(rec(:,:,ind));
        sinogram_tmp = gpuArray(sinogram(ind,:,:));
        cfg_tmp = cfg;
        cfg_tmp.iVolZ = length(ind); 
        cfg_tmp.iProjV = length(ind); 
        cache_tmp = cache;
        cache_tmp.R = cache_tmp.R(ind,:,:); 
        
        %% call normal SART 
        rec_tmp = SART(rec_tmp, sinogram_tmp, cfg_tmp, vectors, cache_tmp, relax ) ; 
        rec(:,:,ind) = gather(rec_tmp);
    end
end
