% Analyse a few projections to determine the sample alignment and wobble
% compensation
% Click first right, then left then height

% In order to run this code save a script with a name such as ptycho_align
% from a working template_ptycho script with the correct initial probe
% and other reconstructions parameters. Then write the name of the script
% in the variable ptycho_template_name. In the ptycho_align script you 
% can set a low number of iterations or a small asize so that the
% alignment is faster.
% Also added the option to display either phase or amplitude during the
% clicking.

clear
close all 

addpath ~/Data10/cxs_software/base/
addpath ~/Data10/cxs_software/ptycho/
addpath ~/Data10/cxs_software/ptycho/utils
addpath ~/Data10/cxs_software/tomo/
addpath ~/Data10/cxs_software/tomo/utils
base_path = '~/Data10/';
analysis_path = fullfile(base_path,'analysis');
spec_internal_path = fullfile(base_path,'specES1/internal/');
import io.*
import utils.*

%%%%%%%%%%%%%%%%%%%%%%%%%
%%%% User parameters %%%%
%%%%%%%%%%%%%%%%%%%%%%%%%
scan_fov = 10e-6;               % Field of view of scan in meters. Ensure the range of the piezos is not exceeded

verbose(1)
omny_direct_scans = true;       % If Orchestra is handling the scans rather than spec
contrast_mode = 'phase';        % Shows either phase or amplitude when clicking ('phase','abs')
ptycho_template_name = 'ptycho_align'; % Remember to change the number of iterations in the ptycho template
single_alignment_file = false;   % If X-ray align values are in a single file, ignores omny_direct_numproj
                                % (= false for OMNY and flOMNI and = true for LaMNI)
vertical_order_correction = 3;  % Harmonic of the vertical correction ( = 3 for OMNY and = 1 for LaMNI) 
one_click_version = false;       % If you just want to click on one feature rather than 3 clicks
%%% LaMNI special %%%
lamni_fit = false;               % For LaMNI instead of a normal fit for ptycho you need to click many 
                                % times and save a special text file that is then used for scanning with LaMNI,
                                % only active when use_ptychography
lamni_fit_file = '';            % For LaMNI there are many clicks to make, so better save them
                                % put here the name of the file. If the file exists it will not ask you to click, it will 
                                % just load them and rewrite the correction file. 
minimal_recommended_gap= 1;     % [microns], recommended gap around the minimum field of view

%%%% Override parameters %%%%  currently needed for LaMNI
force_use_ptychography = []; %true or false
force_scan_numbers     = []; %[271:5:496];

%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%% Read scan numbers %%%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%
screensize = get( 0, 'Screensize' );

if omny_direct_scans
    if ~single_alignment_file
        omny_direct_numproj = 5;        % Assumes the angles are equally distributed in [0,180] degrees                                
        for ii = 0:omny_direct_numproj-1
            filename = fullfile(spec_internal_path,['ptycho_alignmentvalue_x_' num2str(ii)]);
            if ~exist(filename,'file')
                if ii == 0
                    display('Performing alignment based on X-ray eye values')
                end
                use_ptychography = 0;
                filename = fullfile(spec_internal_path,['xrayeye_alignmentvalue_x_' num2str(ii)]);
            else
                use_ptychography = 1;
            end
            fid = fopen(filename);
            aux = fread(fid,'*char')';
            fclose(fid);
            theta = linspace(0,180,omny_direct_numproj);
            opos1(ii+1) = str2double(aux);
            opos0(ii+1) = 0;
        end
    else 
        filename = fullfile(spec_internal_path,['xrayeye_alignmentvalues']);
        if exist(filename,'file')
            fid = fopen(filename);
            outmat = textscan(fid,'%f %f %f','headerlines',1);
            fclose(fid);
            theta = outmat{1}.';
            opos1 = outmat{2}.'; % horizontal
            opos0 = outmat{3}.'; % vetical
            use_ptychography = 0;
            omny_direct_numproj = numel(theta);
        else
            use_ptychography = 1;
        end
    end
end

if force_use_ptychography
    use_ptychography = 1;
    warning('Override force_use_ptychography is active, I will then use ptychography')
end

if use_ptychography
    run(ptycho_template_name)
    if isempty(force_scan_numbers)
        % Read scans from filesystem
        p.plot.prepared_data = false;
        filename = [spec_internal_path 'ptychotomoalign_scannum.txt'];
        fid = fopen(filename);
        scanfirst = fread(fid,'*char')';
        fclose(fid);
        scanfirst = str2num(scanfirst);
        % Read number of projections
        filename = [spec_internal_path 'ptychotomoalign_numproj.txt'];
        fid = fopen(filename);
        numproj = fread(fid,'*char')';
        fclose(fid);
        numproj = str2num(numproj);
        
        display('Alignment using ptychography scans')
        display(['Scan number ' num2str(scanfirst) ', ' num2str(numproj) ' projections'])
        
        if omny_direct_scans
            scans = [scanfirst:scanfirst+numproj-1];
        else
            scans = [scanfirst:2:scanfirst+2*numproj-2];
        end
        
    else
        warning(sprintf('Override scan numbers is active, I will then use the scannumbers in force_scan_numbers'))
        scans = force_scan_numbers;
        numproj = numel(scans);
    end
    display(['Scans to be used ' num2str(scans)])
end



% %% Read angles from SPEC

if ~omny_direct_scans
    for ii = 1:size(scans,2)
        x = spec_read('~/Data10/','ScanNr',scans(ii));
        theta(ii) = x.samroy;
        x2 = spec_read('~/Data10/','ScanNr',scans(ii)+1);
        opos0(ii) = x2.pos1;
        opos1(ii) = x2.pos2;
    end
end


%% 
%%%%%%%%%%%%%%%%%%%%%%
%%%% Ptychography %%%%
%%%%%%%%%%%%%%%%%%%%%%
name = [''];
% Check if the reconstructions exist
% Run a few iterations for reconstruction
if use_ptychography
    for ii = 1:size(scans,2),
        display(['Scan ' num2str(ii)])
        filenameptycho = find_ptycho_filename(analysis_path,scans(ii),name,'_recons'); 
        if iscell(filenameptycho)
            filenameptycho = filenameptycho{1};
        end
        if ~exist(filenameptycho,'file')
            cd ../ptycho
            p.scan_number   = scans(ii);
            p.prefix        = name;
            core.ptycho_recons(p);
        end
    end
end

%% %%% Load reconstructions
if use_ptychography
    if ~exist(lamni_fit_file,'file')
        for ii = 1:size(scans,2)
            %     parfor ii = 1:size(scans,2)
            filenameptycho = find_ptycho_filename(analysis_path,scans(ii),name,'_recons');
            if iscell(filenameptycho)
                filenameptycho = filenameptycho{1};
            end
            fprintf('Loading %s\n',filenameptycho)
            [this_object, probe, p] = load_ptycho_recons(filenameptycho);
            object{ii} = this_object;
        end
        object_size = size(object{1});
    else
        fprintf('Found %s, skipping loading of recons files',lamni_fit_file);
    end
end

%% %%% Start clicking away
if use_ptychography
    if exist(lamni_fit_file,'file')
        aux = load(lamni_fit_file);
        samposx = aux.samposx;
        samposy = aux.samposy;
    else
        for ii = 1:size(scans,2)
            x = zeros(3,1);
            y = zeros(3,1);
            for jj = 1:3
                figure(1)
                if strcmpi(contrast_mode,'phase')
                    imagesc(([1 object_size(2)]-floor(object_size(2)/2)+1)*p.dx_spec(2)*1e6,([1 object_size(1)]-floor(object_size(1)/2)+1)*p.dx_spec(1)*1e6,angle(object{ii}));
                elseif strcmpi(contrast_mode,'abs')
                    imagesc(([1 object_size(2)]-floor(object_size(2)/2)+1)*p.dx_spec(2)*1e6,([1 object_size(1)]-floor(object_size(1)/2)+1)*p.dx_spec(1)*1e6,abs(object{ii}));
                else
                    error(['I don''t know what (contrast_mode = ' contrast_mode ') means. Please choose phase or abs'])
                end
                xlabel('microns')
                ylabel('microns')
                colormap(bone(256));
                title(sprintf('phase: %d', scans(ii)));
                axis image xy tight
                grid on;
                if ~one_click_version
                    display('First click the left, then the right, then the height')
                    if jj >= 2
                        hold on
                        plot([x(1) x(1)],([1 object_size(1)]-floor(object_size(1)/2)+1)*p.dx_spec(1)*1e6,'r')
                        hold off
                    end
                    if jj >= 3
                        hold on
                        plot([x(2) x(2)],([1 object_size(1)]-floor(object_size(1)/2)+1)*p.dx_spec(1)*1e6,'r')
                        plot([(x(2)+x(1))/2 (x(2)+x(1))/2],([1 object_size(1)]-floor(object_size(1)/2)+1)*p.dx_spec(1)*1e6,'--r')
                        hold off
                    end
                    title('First click the left, then the right, then the height')
%                     [x(jj),y(jj)] = ginput(1);
                    [x(jj),y(jj),button_types] = plotting.ginput_ax_mod2(gca, 1);
                else
                    if jj == 1
                        title('One click version')
%                         [samposx(ii),samposy(ii)] = ginput(1);
                        [samposx(ii),samposy(ii),button_types] = plotting.ginput_ax_mod2(gca, 1);
                    end
                end
            end
            if ~one_click_version
                hold on
                plot(([1 object_size(2)]-floor(object_size(2)/2)+1)*p.dx_spec(1)*1e6,[y(3) y(3)],'--g')
                hold off
                pause(0.2)
                samposx(ii) = (x(2)+x(1))/2;
                fov(ii) = abs(x(2)-x(1));
                samposy(ii) = y(3);
            end
        end
        save(lamni_fit_file);
        
        %% at the end show imagesc3D of all projection to help decide about the field of view 
        
        if ishandle(5); close(5); end
        figure(5)
        clf
        for ii = 1:length(scans)
            object{ii} = utils.crop_pad(object{ii}, size(object{1}));
        end
        img = cat(3, object{:}); 
        % apply center of rotation correction 
        if exist('samposy','var')
            img = utils.imshift_fft(img, -(samposx)/(p.dx_spec(1)*1e6),-(samposy)/(p.dx_spec(1)*1e6));
        else
            img = utils.imshift_fft(img, -(samposx)/(p.dx_spec(1)*1e6),0);
        end
        img = utils.crop_pad(img,object_size-p.asize); 
        if strcmpi(contrast_mode,'phase')
            img = angle(img);
        else
            img = abs(img);
        end
        roi_size = object_size-p.asize; 
        plotting.imagesc3D(([1 roi_size(2)]-floor(roi_size(2)/2)+1)*p.dx_spec(2)*1e6,([1 roi_size(1)]-floor(roi_size(1)/2)+1)*p.dx_spec(1)*1e6,img);
        axis image xy tight
        if exist('fov')
            hold on
            plotting.vline(- max(fov)/2, 'r--','Minimal')
            plotting.vline(+ max(fov)/2, 'r--')
            plotting.vline(- max(fov)/2-minimal_recommended_gap, 'b-')
            plotting.vline(+ max(fov)/2+minimal_recommended_gap, 'b-','Recommended')
            hold off
        end
        xlabel('microns')
        ylabel('microns')
        colormap(bone(256));
        title('(Scroll through the projections to see all angles)')
        if exist('fov')
            plotting.suptitle(sprintf('Showing maximal horizontal FOV of %.2gum and full vertical FOV of %.2gum\n Recommended horizontal range is ~ %.2gum+2x%.2gum=%.2gum',  max(fov), floor(roi_size(1))*p.dx_spec(2)*1e6,  max(fov), minimal_recommended_gap,  max(fov)+ 2*minimal_recommended_gap));
        end
        grid on;

        set(gcf,'Outerposition',[1 screensize(4)-1000 800 500]);



    end
else
    samposx(1:omny_direct_numproj) = 0;   
    samposy(1:omny_direct_numproj) = 0;
end


% figure(1)
% subplot(2,1,1)
% plot(theta,opos1,'--o')
% title('horizontal')
% subplot(2,1,2)
% plot(theta,opos0,'--o')
% title('vertical')

% figure(100)
% plot(samposx,samposy,'--o')

%% Fit
if lamni_fit&&use_ptychography
    ellipse_fit_positions(samposx,samposy,scans,lamni_fit_file)
else
    %%% Fit sinusoidal style, good for flOMNI, OMNY and LaMNI (but for the last one only the X-ray eye alignment
    if (max(theta)-min(theta))>270
        sym_flip = false;
    else
        sym_flip = true;
    end
    
    % [theta(theta>=0) theta(theta<0)+360]
    f = fittype('a*sin(x*pi/180+b)+c');
    aux = opos1;
    thetaaux = theta;
    if theta(1)+360 == theta(end)  %% Average repeated angles for laminography
        auxopos1    = opos1(1:end-1);
        auxopos1(1) = opos1(1)/2+opos1(end)/2;
        auxsamposx  = samposx(1:end-1);
        auxsamposx(1) = samposx(1)/2+samposx(end)/2;
        
        auxopos0 = opos0(1:end-1);
        auxopos0(1) = opos0(1)/2+opos0(end)/2;
        auxsamposy = samposy(1:end-1);
        auxsamposy(1) = samposy(1)/2+samposy(end)/2;
        
        thetaaux = theta(1:end-1);
    else
        auxopos1    = opos1;
        auxsamposx  = samposx;
        auxopos0    = opos0;
        auxsamposy  = samposy;
    end
    fitx = fourier_interpolate((auxopos1-auxsamposx)',thetaaux.'*pi/180,1,sym_flip);
    fitx.b = fitx.b - min(theta)*pi/180;
    % fitx = fit(theta',(opos1-samposx)',f,'StartPoint',[0.1 0 mean(opos1-samposx)]),
    % fitx_xrayeyeonly = fit(theta',(opos1)',f,'StartPoint',[0.1 0 mean(opos1)]),
    theta2 = linspace(min(theta),max(theta),200);
    
    % fity = fit(theta',(opos0-samposy)',f,'StartPoint',[1 1 1]*0),
    % fity = fourier_interpolate((opos0-samposy)',theta.'*pi/180);
    fity = fourier_interpolate((auxopos0-auxsamposy)',thetaaux.'*pi/180,vertical_order_correction,sym_flip);
    fity.b(:) = fity.b(:) - min(theta)*pi/180;
    
    auxfunc = fity.c;
    for iii = 1:numel(fity.a)
        auxfunc = auxfunc + fity.a(iii)*sin(iii*theta2*pi/180+fity.b(iii));
    end
    

    figure(1)
    plot(theta,opos1-samposx,'.-')
    title(['Horizontal fit (opos1-samposx)'])
    hold on
    plot(theta2,fitx.a*sin(theta2*pi/180+fitx.b)+fitx.c,'r');
    plot(theta2,fitx.a*sin(theta2*pi/180+fitx.b)+fitx.c+scan_fov*1e6/2,'--r');
    plot(theta2,fitx.a*sin(theta2*pi/180+fitx.b)+fitx.c-scan_fov*1e6/2,'--r');
    hold off
    xlabel('Rotation [degrees]')
    ylabel('microns')
    
    maxscanpos = max(max(abs(fitx.a*sin(theta2*pi/180+fitx.b)+fitx.c+scan_fov*1e6/2),abs(fitx.a*sin(theta2*pi/180+fitx.b)+fitx.c-scan_fov*1e6/2)));
        
    set(gcf,'Outerposition',[1 screensize(4)-500 800 500]);

    
    figure(3)
    plot(theta,opos0-samposy,'.-')
    hold on
    % plot(theta2,fity.a*sin(theta2*pi/180+fity.b)+fity.c,'r');
    plot(theta2,auxfunc,'r');
    hold off
    title(['Vertical fit (opos0-samposy)'])
    xlabel('Rotation [degrees]')
    
    %figure(4)
    %plot(theta,opos0-samposy - fity.a*sin(theta*pi/180+fity.b)+fity.c,'.-')
    
    title(['Vertical fit difference (opos0-samposy)-fit'])
    xlabel('Rotation [degrees]')
    
    set(gcf,'Outerposition',[800 screensize(4)-500 800 500]);




    if use_ptychography
        display(['Maximum FOV = ' num2str(max(fov)) ' microns'])
    end
    
    if maxscanpos < 125
        display(['Maximum piezo displacement, including ' num2str(scan_fov*1e6) ' micron FOV, is ' num2str(maxscanpos) ' microns'])
    else
        warning(['Maximum piezo displacement is too large, including ' num2str(scan_fov*1e6) ' micron FOV, is ' num2str(maxscanpos) ' microns'])
    end
    
    %%
    %%%%%%%%%%%%%%%%%%%%%%%%%%
    %%%% Write parameters %%%%
    %%%%%%%%%%%%%%%%%%%%%%%%%%
    
    answ = questdlg('Was the alignment OK? Click "no" if fine alignment scans need to be repeated','Plot costumer support',...
        'Yes, awesome','No','No');
    switch answ
        case 'Yes, awesome'
            %         msgbox('Writting these values to file')
            writefiles = 1;
        otherwise
            msgbox('Oops, then I don''t write the values to file')
            writefiles = 0;
    end
    if writefiles
        % Write parameters in files
        filename = '~/Data10/specES1/internal/ptychotomoalign_Ax.txt';
        fid = fopen(filename,'w');
        fprintf(fid,'%f',fitx.a);
        fclose(fid);
        filename = '~/Data10/specES1/internal/ptychotomoalign_Bx.txt';
        fid = fopen(filename,'w');
        fprintf(fid,'%f',fitx.b);
        fclose(fid);
        filename = '~/Data10/specES1/internal/ptychotomoalign_Cx.txt';
        fid = fopen(filename,'w');
        fprintf(fid,'%f',fitx.c);
        fclose(fid);
        filename = '~/Data10/specES1/internal/ptychotomoalign_Ay.txt';
        fid = fopen(filename,'w');
        fprintf(fid,'%f',fity.a(1));
        fclose(fid);
        filename = '~/Data10/specES1/internal/ptychotomoalign_By.txt';
        fid = fopen(filename,'w');
        fprintf(fid,'%f',fity.b(1));
        fclose(fid);
        filename = '~/Data10/specES1/internal/ptychotomoalign_Cy.txt';
        fid = fopen(filename,'w');
        fprintf(fid,'%f',fity.c);
        fclose(fid);
        display('Alignment parameters have been written to file')
        %%% For OMNY we now use the third harmonic of the position vs theta
        if numel(fity.a)>2
            filename = '~/Data10/specES1/internal/ptychotomoalign_Ay3.txt';
            fid = fopen(filename,'w');
            fprintf(fid,'%f',fity.a(3));
            fclose(fid);
            filename = '~/Data10/specES1/internal/ptychotomoalign_By3.txt';
            fid = fopen(filename,'w');
            fprintf(fid,'%f',fity.b(3));
            fclose(fid);
        end
    end

end


% % Send to EPICS
% scall = sprintf('caputq X12SA-ES1-DOUBLE-03 %d',fity.c);
% [err,io] = system(scall);
% % % scall = sprintf('caputq X12SA-ES1-DOUBLE-04 %d',fity.b);
% % % [err,io] = system(scall);
% % % scall = sprintf('caputq X12SA-ES1-DOUBLE-05 %d',fity.);
% % % [err,io] = system(scall);
% scall = sprintf('caputq X12SA-ES1-DOUBLE-04 %d',fitx.a);
% [err,io] = system(scall);
% scall = sprintf('caputq X12SA-ES1-DOUBLE-05 %d',fitx.b);
% [err,io] = system(scall);
% scall = sprintf('caputq X12SA-ES1-DOUBLE-06 %d',fitx.c);
% [err,io] = system(scall);


return;





