% template_tomo_recons.m
% to process ptychographic reconstructions from many projections
% and make tomographic reconstruction

% This code and subroutines are part of a continuous develpment. There is no 
% liability on PSI or cSAXS. Before publishing results using this code 
% please notify M. Guizar-Sicairos, (mguizar@gmail.com)
% or another PSI cSAXS staff member.

% Publications most relevant to this code
% M. Guizar-Sicairos, A. Diaz, M. Holler, M. S. Lucas, A. Menzel, R. A. Wepf, and O. Bunk
% "Phase tomography from x-ray coherent diffractive imaging projections," Opt. Express 19, 21345-21357 (2011).
% For tomographic consistency alignment:
% M. Guizar-Sicairos, J. J. Boon, K. Mader, A. Diaz, A. Menzel, and O. Bunk,
% "Quantitative interior x-ray nanotomography by a hybrid imaging technique," Optica 2, 259-266 (2015).



% NOTE: ONLINE VERSION - you can run online version using 'tomo_recons_online.m' template
% prevent unwanted calling of the template 
if length(dbstack) == 1   
     return 
end

close all 

%% Add paths
base_path='../../';
addpath('utils')
addpath('tests')
addpath(find_base_package())



import plotting.*
import io.*
if length(dbstack) == 1  
    debug(false) % keep debug option false in case of manual run 
end

%% File management
if debug() <= 1  % control point for the automatic code tests
%%%%%%%%%%%%%%%%%%%%%%%%%
%%% Edit this section %%%
%%%%%%%%%%%%%%%%%%%%%%%%%

par.scanstomo = [];
par.tomo_id = 153; % Either scan numbers or tomo_id can be given, but not both, if not provided leave tomo_id=[]

% verbosity and online tomo 
par.online_tomo = length(dbstack)>2;   % automatically run if called from externally
par.verbose_level = 1; 
par.base_path=base_path;
par.showrecons = par.verbose_level > 1;

% IO loading 
par.fileprefix='';                  % string at the beginning of the filename, related to reconstruction name
par.filesuffix = '400x400_run_1_recons';   % string searched at the end of the data filenames,  No need to add _c or _recons, it will look for it 
par.file_extension = 'h5';
par.downsample_projections = 0;     % downsample projections by factor of 2^x, set 0 to do nothing and 1,2,.. for different levels of projection binning 

par.use_mex_loader = true;   % if true, use the new fast MEX-based loader, false is rather a backup option
par.Nthreads_mexread = min(10,feature('numcores'));  % number of threads for loading, try to set 6 if you see some failure in the mex_read binary 
par.analysis_path = [par.base_path 'analysis/'];      % path from where projections are loaded 
par.output_path = [par.base_path 'analysis_tomo/'];   % path where the tomograms are stored 
par.clip_amplitude_quantile = 0.95;  %  clip amplitudes in the loaded projections that are exceeding given quantile, if par.clip_amplitude_quantile == 1, do nothing 
par.max_residua_limit = 100;        % limit used to determine which projection have failed 
par.sort_by_angle = false;          % If true projections are sorted by angle rather than by the scan numbers, this sorting can effect alignment 

% IO saving
par.output_folder_prefix = '';      % a string that will be attached to the output folder name in form tomo_+output_folder_prefix+autogenerated_description
par.save_final_data = true; 
par.force_overwrite = par.online_tomo ;     % ask before overwritting data 
par.save_temp_data = false;      % 1 to save temporal data to disk, ! final results are saved always !

% Angles + auto calibration 
par.use_OMNY_file_angles = 1;        %% Read angles from special file in OMNY, otherwise it uses SPEC dat file
par.checkangles = false;             %% Compares spec angles with expected
par.auto_alignment = 0;     %% Uses interferometer positions to prealign projections - for now it will read them from the spec file. 
			% This feature needs a new calibration of the sphere and is not fully tested.
par.get_auto_calibration = 1;  % If you want to use this recons to get a calibration file for autotomo  
par.remove_duplicated_angles = true; % remove angles measured twice, should be true for FBP method 
par.omnyposfile = [par.base_path 'specES1/scan_positions/scan_%05d.dat'];		%Filename pattern for Orchestra interferometer position files   
par.OMNY_angle_file = [par.base_path, 'specES1/dat-files/tomography_scannumbers.txt'];  % Filename with angles

% Other
par.showsorted = true;      % sort plotted projections by angles, it has not effect on the alignment itself 
par.windowautopos = true;
par.save_memory = false;        % try to limit use of RAM 
    par.inplace_processing = par.save_memory; % process object_stack using inplace operations to save memory 
    par.fp16_precision     = par.save_memory; % use 16-bit precision to store the complex-valued projections 
    par.cache_stack_object = par.save_memory; % store stack_object to disk when no needed 



par.GPU_list = [1];     % number of the used GPU % If you want to check usage of GPU 
                        % > nvidia-smi
                        % Then in matlab use  par.GPU_list = 2  for example to use the second GPU 
par.Nworkers = min(10,feature('numcores'));  % number of workers for parfor. avoid starting too many workers


%%%% SET TRUE FOR LAMINO AND FALSE FOR MISSING WEDGE TOMO 
par.is_laminography = true;         % false -> standard tomography / limited angle tomography
                                    % true = allow some specific options for laminography reconstruction, ie circular field of view ...



if par.is_laminography
    %%%% geometry settings laminography %% 
    %  Geometry estimation: 
    %   a) initial guess has to be provided from known geometry, 
    %           lamino_angle is the smallest angle between sample rotation plane and beam
    %           tilt_angle  is a rotation around the beam axis that makes the sample rotation axis parallel with vertical axis of camera 
    %           skewness_angle - angle between vertical and horizontal axis of the projection, Im not sure why it is not zero
    %   b) improved estimate from tomo.align_tomo_global_parameters
    %       simple line search that shows evolution of total variation, contrast and sparsity in dependence on a selected geometry parameter 
    %   c) self-consistent estimation during alignment 
    %       set par.refine_geometry = true; but the sample should be already well aligned and the initial guess of geometry has to be rather close 
    %
    % default settings for Lamni
    par.lamino_angle   = 61.108;        % laminography angle, should be 90 for common tomo
    par.tilt_angle     = -72.605;      % rotation of the camera around the beam direction, should be 0 for common tomo 
    par.skewness_angle = 1.296;        % shear of the "camera axis" , should be 0 for common tomography

else
    % otherwise assume missing wedge tomography 
    par.lamino_angle = 90.0;         % laminography angle, should be 90 for common tomo
    par.tilt_angle = 0.0;            % rotation of the camera around the beam direction, should be 0 for common tomo 
    par.skewness_angle = 0.0;        % shear of the "camera axis" , should be 0 for common tomography
end
par.vertical_scale = 1;             % relative pixel scale between the projection and reconstruction (ie voxel size and the pixel size in projection does not need to be 1:1)
par.horizontal_scale = 1;           % relative pixel scale between the projection and reconstruction (in if vertical_scale ~= horizontal_scale then the pixels in projection are not square)
Npix_projection = [];               % expected size of the projection, it is useful to create some padding around the projections to avoid artefacts after rotation of the projections 
%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%

end

if debug() == 1 % control point for the automatic code tests
    return
end

block_fun_cfg = struct('GPU_list', par.GPU_list, 'inplace', par.inplace_processing); 

if all(debug() ~= [1,2]) % ignore during automatic tests 

%%% load angles for the projections 
[par,theta] = prepare.load_angles(par, par.scanstomo, par.tomo_id);
theta = - theta;   % definition of angle direction is different in LAMNI
par.scans = par.scanstomo;

%%% prepare preprocessing function for the projections 
if par.is_laminography
    % default for LAMNI 
    object_preprocess_fun = @(x)(utils.crop_pad(utils.imshear_fft(utils.imrotate_ax_fft(x,-par.tilt_angle),-par.skewness_angle,1), Npix_projection));  % negative sign is important to keep the positive angular direction identical with ASTRA definition 
    par.tilt_angle = 0; % loaded projections are already rotated, so the tilt angle should not be used by ASTRA 
    par.skewness_angle = 0; % projections will be already skewed when loaded
else
    % in case of missing wedge tomo, default is no preprocessing 
    object_preprocess_fun = @(x)x;
end

if debug() == 3 
    % Lamni chip dataset needs to be propagated 12 microns before
    % reconstruction, -> perform the propagation already during data
    % loading
    propagation_distance = 12e-6; 
    pixel_size = 13.02e-9;  % other values needs to be hardcoded because they are not yet loaded at this point
    lambda = 0.2e-9; 
    object_preprocess_fun = @(x)utils.prop_free_nf( object_preprocess_fun(x) , lambda, propagation_distance, pixel_size); 
end



%%% perform initial checks 
[par, angles_check, object] = prepare.initialize_tomo(par, par.scanstomo, true, object_preprocess_fun); 



%% Reads ptychographic reconstructions and stores them in stack_object
%%%%%%%%%%%%%%%%%%%%%%%%%
%%% Edit this section %%%
%%%%%%%%%%%%%%%%%%%%%%%%%
exclude_scans = [];
%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%

[stack_object, theta, ~, par] = prepare.load_projections_fast(par,exclude_scans,par.dims_ob,theta,object_preprocess_fun);
 
end


%% apply some very preliminary phase ramp removal - has to be applied before the dataset is shifted to avoid ramp artefacts at the edges 
utils.verbose(-1,'Roughly remove phase-ramp and normalize amplitude')
stack_object = tomo.block_fun(@utils.stabilize_phase,stack_object, 'weight', par.illum_sum / max(par.illum_sum(:)),'normalize_amplitude',true, block_fun_cfg);



%% Display reconstructed phases
%%%%%%%%%%%%%%%%%%%%%%%%%
%%% Edit this section %%%
%%%%%%%%%%%%%%%%%%%%%%%%%
par.baraxis = 'auto';       % = 'auto'  or   = [-1 1]
par.windowautopos = true;  % automatic placemement of the plot
par.showsorted = true;      % sort the projections by angles 
%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%
if ~par.online_tomo || debug()
    tomo.show_projections(stack_object, theta, par, 'fnct', @angle, 'title', 'Full original projections before alignmnent','plot_residua', false,'fps',2)
end

%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%  RECONSTRUCTION %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

%%%%%%%%%%%%%%%%%%%%%%%%%
%%% Edit this section %%%
%%%%%%%%%%%%%%%%%%%%%%%%%
[Nx,Ny,Nangles] = size(stack_object);

% Choose reconstructed region
% default is take as much as possible 
total_shift = zeros(Nangles,2);


reduce_fun = @max; % select the the field of view in at least one projection 

reduced_proj = tomo.block_fun(@abs, stack_object, struct('use_GPU', false, 'reduce_fun', reduce_fun)); 
weight_sino = gpuArray(single(reduced_proj > 0.1*max(reduced_proj(:)))); 
weight_sino = convn(weight_sino, ones(par.asize)/(prod(par.asize)), 'same'); 
weight_sino = gather( utils.imgaussfilt2_fft(weight_sino > 0.5, 10)); 
if  all(all(all(weight_sino == mean(weight_sino,3))))
    weight_sino = weight_sino(:,:,1);  % if all are the same, store only the first 
end
[object_ROI] = get_ROI(weight_sino>0.5, 0);
[object_ROI_xcorr] = get_ROI(weight_sino>0.5, -0.15, 32); % Changed the second entry to optimise ROI so that it doesn't contain the noise in the corners

object_ROI = {intersect(ceil(1+par.asize(1)/2:Nx-par.asize(1)/2), object_ROI{1}),...
              intersect(ceil(1+par.asize(2)/2:Ny-par.asize(2)/2), object_ROI{2})};
          
par.air_gap = [];           % very roughly distance around sides where is assumed air 


%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%

% Make data easily splitable for ASTRA
width_sinogram = floor(length(object_ROI{2})/64)*64;
Nlayers = floor(length(object_ROI{1})/64)*64;
Nroi = [length(object_ROI{1}),length(object_ROI{2})];
object_ROI = {object_ROI{1}(ceil(Nroi(1)/2))+[1-Nlayers/2:Nlayers/2],...
      object_ROI{2}(ceil(Nroi(2)/2))+[1-width_sinogram/2:width_sinogram/2]}; 

%% Display reconstructed phases
%%%%%%%%%%%%%%%%%%%%%%%%%
%%% Edit this section %%%
%%%%%%%%%%%%%%%%%%%%%%%%%
par.baraxis = 'auto';       % = 'auto'  or   = [-1 1]
par.windowautopos = true;  % automatic placemement of the plot
par.showsorted = true;      % sort the projections by angles 
plot_residua = false;      % if true, denote residua in projections by red circles. 

%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%
tomo.show_projections(stack_object, theta, par, 'fnct', @angle, ...
    'title', 'Full original projections before alignmnent','plot_residua', plot_residua, ...
    'rectangle_pos', [object_ROI{2}(1), object_ROI{2}(end), object_ROI{1}(1), object_ROI{1}(end)]) 

 
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%% Manual removal of poor projections %%%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
which_remove =  [];         % list of projection indices or bool vector of projections to be removed  
                            % Examples: 
                            %      which_remove = [1,5,10]
                            %      which_remove = ismember(par.scanstomo, [264,1023])
                            %      which_remove = theta > 0.5 & theta < 15
                            %       
                            %      
plot_fnct = @angle;         % function used to preprocess the complex projections before plotting, use @(x)x to show raw projections 

[stack_object,theta,total_shift,par] = tomo.remove_projections(stack_object,theta,total_shift,par, which_remove, plot_fnct, object_ROI); 
[Nx,Ny,Nangles] = size(stack_object);



%% Cross-correlation alignment of raw data - only rough guess to ease the following steps 
%%%%%%%%%%%%%%%%%%%%%%%%%
%%% Edit this section %%%
%%%%%%%%%%%%%%%%%%%%%%%%%
par.filter_pos = 101;        %  highpass filter on the evolution of the recovered positions applied in the following form:  X - smooth(X,par.filter_pos), it prevents accumulation of drifts in the reconstructed shifts 
par.filter_data = 0.001;    %  highpass filter on the sinograms to avoid effects of low spatial freq. errors like phase-ramp 
par.max_iter = 10;           %  maximal number of iterations 
par.precision = 0.1;          %  pixels; stopping criterion
par.binning = 4;           %  binning used to speed up the cross-correlation guess 
eroding_radius = [];   % Erodes FOV in order to choose the ROI. Leave empty for default max(par.asize)/2
%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%

if isempty(eroding_radius)
    eroding_radius = max(par.asize)/2;
end

utils.verbose(-1,'Cross-correlation pre-alignment of raw data')
object_ROI = {object_ROI{1}(ceil(Nroi(1)/2))+[1-Nlayers/2:Nlayers/2],...
      object_ROI{2}(ceil(Nroi(2)/2))+[1-width_sinogram/2:width_sinogram/2]}; 

% estimate smaller ROI for alignment 
xcorr_ROI = get_ROI(imerode(utils.Garray(par.illum_sum > 0.1), strel('disk', eroding_radius))); 
[xcorr_shift, variation_binned, variation_aligned]  = tomo.align_tomo_Xcorr(stack_object, theta, par, 'ROI', xcorr_ROI);

if ~par.online_tomo || debug()
    % show prealigned projections 
    tomo.show_projections(cat(1,variation_aligned, variation_binned), theta, par, 'title', sprintf('Preview of cross-correlation pre-aligned projections, binning %ix \n Top: original Bottom: aligned \n', par.binning), 'figure_id', 12)
    axis off image 
end
clear variation_binned variation_aligned

if par.online_tomo || debug() || ~strcmpi(input('Apply shifts found by cross-correlation: [Y/n]\n', 's'), 'n')
    % shift complex projections, use circular shift to avoid information loss 
    stack_object = tomo.block_fun(@utils.imshift_linear, stack_object, xcorr_shift(:,1),xcorr_shift(:,2), 'circ', block_fun_cfg);
    total_shift = total_shift + xcorr_shift;
end


%% TOMOCONSISTENCY ALIGNMENT
%%%%%%%%%%%%%%%%%%%%%%%%%
%%% Edit this section %%%
%%%%%%%%%%%%%%%%%%%%%%%%%


%%%%%%%%%% TUNABLE PARAMETERS %%%%%%%%%%%%%%%%%%%%

%% ESTIMATE THICKNESS OF THE SAMPLE -> used to set size of the reconstructed volume 

% ::IMPORTANT:: SET HERE THE EXPECTED SAMPLE THICKNESS (only of the scattering layers)
sample_thickness =  10e-6; 
method_for_reliability_region = 'quantile';  %'quantile' uses quantile on the whole stack to find a common threshold for all projections
                                             %'morphological' uses Otsu threshold on each projection, and additionally uses morphological operations to refine the FOV
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

% settings for automatic tests, do not change 
if debug()==2
    sample_thickness = 10e-6; % optimal value for the synthetic tests 
elseif debug()==3 
    sample_thickness = 7e-6;  % set optimal thickness for lamni large chip dataset 
end
warning  off backtrace; warning('Assuming sample thickness of %gum', sample_thickness*1e6); warning  on backtrace
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

%%%%%%%%%% Other parameters %%%%%%%%
max_binning = 2^ceil(log2(max(width_sinogram))-log2(100));   % prevent binning to make smaller object than 100 pixels, reconstruction gets unstable 
                                                              % too large binning may lead to poor results and too small binning will prevent escaping local minima
min_binning = min(max_binning, 2^max(par.online_tomo, debug()>0));

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

% If you want manual region selection/clicking, leave it empty:
definedROI{1} = []; % e.g. [258:534]; % y-range
definedROI{2} = []; % e.g.[ 405:779]; % x-range

par.high_pass_filter = 0.01;        % remove effect of residuums/bad ptycho convergence , high value may get stuck in local minima 
par.showsorted = true;              % selected angular order for alignment 
par.valid_angles = 1:Nangles > 0;   % use only this indices to do reconstruction (with respect to the time sorted angles)
par.align_horizontal = true;        % horizontal alignment 
par.align_vertical = true;         % vertical alignment
par.use_mask = false;               % apply support mask 
par.mask_threshold = 0.001;         % []; % empty == Otsu thresholding 
par.use_localTV = false;            % apply local TV 
par.apply_positivity = false;        % remove negative values 
par.min_step_size  = 0.01;          %stoppig criterion ( subpixel precision )
par.max_iter = 500;                 % maximal number of iterations
par.use_Xcorr_outlier_check = false; % in the first iteration check and remove outliers using Xcorr 
par.center_reconstruction = false;   % keep the center of mass in center of rec. volume 
par.plot_results_every = 20;        % plot results every N seconds

%%% Allow here for automatic geometry refinement 
refine_geometry = false;
par.refine_geometry_parameters = {'shear_angle', 'tilt_angle', 'lamino_angle'};   % list of parameters to be refined : {'shear_angle', 'tilt_angle', 'lamino_angle'}
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
if debug() ==2  %  (tested parameters for synthetic data tests -> debug==2)
    refine_geometry = true; 
end

%%%  Internal parameters, do not change %%%%%%%%
par.step_relaxation = 0.3;          % gradient decent step relaxation, (1 == full step), may be needed to avoid oscilations  
par.filter_type = 'ram-lak';        % FBP filter (ram-lak, hamming, ....)
par.freq_scale = 1;                 % Frequency cutoff
par.unwrap_data_method = 'none';
par.air_gap = [width_sinogram/4,width_sinogram/4];   % just avoid ramp from the projections 
% targeted center of rotation 
param.center_of_rotation = [Nx,Ny]/2;
%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%

if par.cache_stack_object && ~exist('stack_object', 'var')
    stack_object = load_stored_object('stack_object'); 
end


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%% SELECT FIELD OF VIEW FOR ALIGNMENT%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

min_proj = tomo.block_fun(@(x)min(abs(x),[],3),stack_object,struct('use_GPU', false, 'reduce_fun', @min, 'ROI', {object_ROI} )); 
max_ang_proj = tomo.block_fun(@(x)max(angle(x),[],3),stack_object,struct('use_GPU', false, 'reduce_fun', @max, 'ROI', {object_ROI})); 

f = plotting.smart_figure(2);
clf()
imagesc(object_ROI{2},object_ROI{1},min_proj .* max_ang_proj)
hold on 
axis image xy
colormap bone
grid on 
hold on; plot(param.center_of_rotation(2), param.center_of_rotation(1),'or'); hold off 
hold on; plot(mean(object_ROI{2}),mean(object_ROI{1}),'xy'); hold off 
drawnow

if par.online_tomo || debug()
    selected_ROI = object_ROI ;   
    title('Auto-selected region: showing reduced projection for all angles')
elseif isempty(definedROI{1})
    utils.verbose(0,'Manually select region to align ... it is actually possible to select all, including regions outside the measurement FOV ')
    title('Select region for alignment: showing reduced projection for all angles')

    rect = round(getrect);
    rect(1:2) = max([object_ROI{2}(1),object_ROI{1}(1)], rect(1:2));
    rect(3:4) = min(rect(3:4),[length(object_ROI{2}),length(object_ROI{1})]); 
    rect(3:4) = floor(rect(3:4)/32)*32; 

    selected_ROI = {(rect(2):rect(2)+rect(4)-1), (rect(1):rect(1)+rect(3)-1)};
    
    utils.verbose(0,'===========================')
    utils.verbose(-1,'Selected subregion: {%i:%i,%i:%i}\n',rect(2), rect(2)+rect(4), rect(1), rect(1)+rect(3));
    utils.verbose(0,'===========================')
    close(f)
else
    numelout1 = round(numel(definedROI{1})/16)*16;
    definedROI{1} = definedROI{1}(1)+[0:numelout1-1];
    numelout2 = round(numel(definedROI{2})/16)*16;
    definedROI{2} = definedROI{2}(1)+[0:numelout2-1];
    selected_ROI = {definedROI{1}, definedROI{2}};
end

% empirical limit for the minimal binning to avoid crashes due to low
% memory during the alignment 
clear sinogram  tomogram_delta 
min_binning = max(min_binning, 2^nextpow2(ceil(sqrt((length(selected_ROI{1})*length(selected_ROI{2})*Nangles*(64)) / (utils.check_available_memory*1e6)))));

  
% get reliability region for phase unwrapping 
if debug() ~= 2
     weight_sino = tomo.block_fun(@lamino.estimate_reliability_region,stack_object, par.asize, 16, method_for_reliability_region);
     figure(2022); plotting.imagesc3D(weight_sino); axis xy equal tight
else
    weight_sino = [] ;
end

utils.verbose(-1,'Get phase ')
% solve the function blockwise on GPU 

% unwrap the complex phase 
sinogram = -tomo.unwrap2D_fft2_split(stack_object, par.air_gap,1,weight_sino, par.GPU_list, selected_ROI);

if ~par.online_tomo
    % show prealigned projections 
    tomo.show_projections(sinogram, theta, par, 'title', 'Selected region - unwrapped phase','fps',2)
end
if par.cache_stack_object
    utils.savefast_safe('stack_object', 'stack_object', 'par', 'theta','total_shift', true)
    clear stack_object
end

alignment_ROI = {1:length(selected_ROI{1}),1:length(selected_ROI{2})};

% calculate offset of CoR for selected ROI with respect to the object_ROI range 
CoR_offset = param.center_of_rotation(2) - ...
            (length(alignment_ROI{2})/2 -0.5 + ... 
                 alignment_ROI{2}(1)-1 + ...
                   selected_ROI{2}(1)-1);  

%%%%%%%%%%%%%%%%%%%%%%%%%
%% % Edit this section %%
%%%%%%%%%%%%%%%%%%%%%%%%%

if par.is_laminography
    % in case of laminography 
    Npix_align = ceil(0.5/cosd(par.lamino_angle-0.01)*length(alignment_ROI{2}));  % for pillar it can be the same as width_sinogram
    Npix_align = ceil([Npix_align, Npix_align, sample_thickness /par.pixel_size]/32)*32; 
else
    % limited angle tomo 
    Npix_align = ceil(width_sinogram/32)*32;  % for pillar it can be the same as width_sinogram
    % assume asymmetric sample !!
    Npix_align = [Npix_align/4, Npix_align, Nlayers];
end


%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%

%% Check the geometry - IT IS ONLY A BACKUP OPTION IN CASE THAT THE INITIAL GUESS FOR GEOMETRY (MAINLY CENTER OF ROTATION) IS TOO POOR
% find center of rotation or lamino angle or tilt of the projections 
%
%  NOTE:     you can improve quality of this step if you run this step on
%  already aligned dataset, once the reconstruction is good enough, 
%  you can use         par.refine_geometry = true to refine geometry by
%  aling_tomo_consistency_linear function 
%
% 1) select the parameter to be optimized: 
%       search_parameter: 'center_of_rotation', 'center_of_rotation_z',
%                         'lamino_angle', 'tilt_angle', 'rot_angle', 'shear_angle'
% 2) select the search range for a simple grid search method: 
%       'search_range',  [-min_value,max_value]. The other (nonsearched) variables xxx should be set
%       by their xxx_offset values (see example lower) to your best guess (default is 0)
%      the xxx_offset values are during search added to the respective values in the par-structure
%       the par. structure , not that search range is relative for the
%       selected binning 
%      example of the search grid:
%             tilt_angle = par.tilt_angle + tilt_offset + linspace(search_range(1), search_range(2),num_grid_points ); 
% 
% 3) run and visually check the optimal parameter, usually there is a clear
%       maximum / minimum in all plotted parameters
% 4) Are you finished? :
%         YES: MANUALLY  ADD the best value into the par structure and continue to the next steps 
%         NO:  use the ???_angle_offset variables to set the refined
%         parameter, choose another parameter to tune and continue to step 1)
%

if debug() == 2
    % do not run this section unless standard alignment fails 
    utils.verbose(-1,'Semimanual search for geometry parameters')
%     sino_tmp  = tomo.block_fun(@(x)utils.imrotate_ax_fft(utils.imshear_fft(x,-0.6,1), -0.1,3), sinogram); 
    tomo.align_tomo_global_parameters(sinogram,theta(:), Npix_align, par, 'binning', 4, 'selected_roi', alignment_ROI, ...
    'search_range',  [-1, 1] , 'num_grid_points', 20, ...
    'search_parameter',  'tilt_angle', ... %% choose the searched parameter
     'CoR_offset', CoR_offset, ...
     'lamino_angle_offset', 0, ...
     'tilt_angle_offset', 0., ...
     'shear_angle_offset', 0., ...
     'showed_layer_id', []);  % []  means show middle layer 
%% manually set here the user estimated corrections if required 
%   CoR_offset         = CoR_offset         +  [user_selected_offset]  
%   par.lamino_angle   = par.lamino_angle   +  [user_selected_offset];    % laminography angle, should be 90 for common tomo
%   par.tilt_angle     = par.tilt_angle     +  [user_selected_offset];    % rotation of the camera around the beam direction, should be 0 for common tomo 
%   par.skewness_angle = par.skewness_angle +  [user_selected_offset];    % skewness of the "camera axis" , should be 0 for common tomom 

end

%% get at least some initial alignment guess 
shift = zeros(Nangles,2); 

%% 1) start with a robust and slow convergence to get closer to real data ->  horizontal alignment only (more robust)

par.high_pass_filter = 0.01;        % remove effect of residuums/bad ptycho convergence , high value may get stuck in local minima 
par.max_iter = 100; 
par.binning = max_binning; 
par.momentum_acceleration = false; 
par.step_relaxation = 0.1;    
par.center_reconstruction = false;
par.position_update_smoothing = 0.1;      % enforce smoothness of the updates, useful in the initial alignment, 0 = none, 1 = maximal smoothing

par.align_horizontal = true;         % 
par.align_vertical = false;          % skip vertical alignment, horizontal one is more robust 
par.plot_results_every = 10;

utils.verbose(-1,'Horizontal alignment \n');
% self consitency based alignment procedure based on the ASTRA toolbox 
[shift, par] = tomo.align_tomo_consistency_linear(sinogram,weight_sino, theta+0.1, Npix_align, shift, par, 'selected_roi', alignment_ROI, 'CoR_offset', CoR_offset); 

if ~par.online_tomo
    % show aligned projections before shifting full stack_object
    tomo.show_projections(tomo.block_fun(@utils.imshift_fft, sinogram, shift, struct('GPU_list', par.GPU_list)), theta, par,'fps',2,...
        'title', 'Projections after self-consistent alignment')
end


%% 2) if the results are ok, continue also with vertical alignment 

par.align_vertical = true;            % now skip horizontal alignment 
par.align_horizontal = false;         % and allow only vertical alignment search  
par.momentum_acceleration = false; 

par.binning = max_binning; 
par.max_iter = 50; 

utils.verbose(-1,'Vertical alignment ');
[shift, par] = tomo.align_tomo_consistency_linear(sinogram,weight_sino, theta+0.1, Npix_align, shift, par, 'selected_roi', alignment_ROI, 'CoR_offset', CoR_offset); 

if ~par.online_tomo
    % show aligned projections before shifting full stack_object
    tomo.show_projections(tomo.block_fun(@utils.imshift_fft, sinogram, shift, struct('GPU_list', par.GPU_list)), theta, par,'fps',2,...
        'title', 'Projections after self-consistent alignment')
end

%% 4) once a good initial guess of the shift is known, start with alignment in both direction for all binning levels 

% ALIGN FINAL
reset(gpuDevice)
% geometry can be refined once the projections are well prealigned 
interactive_alignment = false;   % If true, before each binning the code will stop and you can have a chance to inspect the projections and to inspect the position. To continue press the "continue" button in Matlab

par.align_vertical = true;           % vertical alignment, usually only a small correction of initial guess  
par.align_horizontal = true;         % horizontal alignment, usually only a small correction of initial guess  
par.position_update_smoothing = 0;   % avoid smoothing of the results 
par.use_Xcorr_outlier_check = true; 
par.step_relaxation = 0.1;    
par.max_iter = 500; 
par.momentum_acceleration = 1;
par.showsorted = 0;

binning = 2.^(log2(max_binning):-1:log2(min_binning)); 
utils.verbose(-1,'Full alignment ');

for jj = 1:length(binning)
    if interactive_alignment
        figure(2)
        plot(shift,'.-')
        tomo.show_projections(tomo.block_fun(@utils.imshift_fft, sinogram, shift, struct('GPU_list', par.GPU_list)), theta, par,...
            'title', 'Projections after self-consistent alignment','fps',2)
        keyboard
    end
    par.binning = binning(jj);
    utils.verbose(-1,'Binning %i ', par.binning);
    
    % do not refine geometry in the lowest binning levels (too misaligned and coarse resolution) and in the two highest levels (too slow)
    if  par.binning > 2 && par.binning < 16  && refine_geometry
        par.refine_geometry = true;
        par.min_step_size  = 1e-3; 
    else  % return original values and 
        par.refine_geometry = false;
        par.min_step_size  = 1e-2; 
    end

    % self consitency based alignment procedure based on the ASTRA toolbox 
    [shift, par, ~, err] = tomo.align_tomo_consistency_linear(sinogram,weight_sino, theta+0.1, Npix_align, shift, par, 'selected_roi', alignment_ROI, 'CoR_offset', CoR_offset); 

    % try to use xcorr to prevent trapping in local minima
    if  par.binning == max_binning && par.use_Xcorr_outlier_check
        xcorr_shift = tomo.align_tomo_consistency_Xcorr(sinogram, weight_sino, theta+0.1, shift, par.binning, Npix_align, par, 'selected_roi', alignment_ROI, 'CoR_offset', CoR_offset); 
        if par.online_tomo || all(abs(xcorr_shift(:)) < 2*par.binning) || strcmpi(input('Apply shifts found by Xcorr-consistent alignment: [y/N]\n', 's'), 'y')
            shift = shift + xcorr_shift ;
        end
    end
    
    % plot the estimated shifts 
    plotting.smart_figure(25)
    clf()
    [~,ind_sort] = sort(theta);
    plot(theta(ind_sort), shift(ind_sort,:), '.')
    legend({ 'Horizontal shift', 'Vertical shift'})
    xlabel('Angle')
    ylabel('Shift [px]')
    xlabel('Sorted angles')
    title('Total shift from self-consistency alignment')
    axis tight ; grid on 
    drawnow
    
    if refine_geometry
        fprintf('=====  Geometry refinement =====\n')
        fprintf('  par.lamino_angle = %d; \n  par.tilt_angle = %d; \n  par.skewness_angle = %d;\n',par.lamino_angle,par.tilt_angle, par.skewness_angle);
    end

end
utils.verbose(-1,'Full alignment done ');


%%
figure(2)
plot(shift,'.-')

if ~par.online_tomo
    % show aligned projections before shifting full stack_object
    tomo.show_projections(tomo.block_fun(@utils.imshift_fft, sinogram, shift, struct('GPU_list', par.GPU_list)), theta, par,...
        'title', 'Projections after self-consistent alignment','fps',2)
end
%%


%% 
clear sinogram rec sino_weights
if par.cache_stack_object && ~exist('stack_object', 'var')
    stack_object = load_stored_object('stack_object'); 
end
if par.online_tomo || debug() || ~strcmpi(input('Apply shifts found by self-consistent alignment: [Y/n]\n', 's'), 'n')
    % shift the complex objects  
    stack_object = tomo.block_fun(@utils.imshift_fft, stack_object, shift, block_fun_cfg);
    % store the total shift
    total_shift = total_shift + shift; 
else
    return
end
if par.cache_stack_object
    utils.savefast_safe('stack_object', 'stack_object', 'par', 'theta','total_shift', true)
end


%%%%%%%%%%%%%%%%%%%%%%%%%%%
% End of alignment 
%%%%%%%%%%%%%%%%%%%%%%%%%%%

%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%% Phase ramp removal + amplitude calibration %%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%% Edit this section %%%
%%%%%%%%%%%%%%%%%%%%%%%%%
Niter = 3;          % number of iterations of phase ramp refinement 
binning = max_binning;      % bin data before phase ramp removal (make it faster)
%%%%%%%%%%%%%%%%%%%%%%%%%%%

par.unwrap_data_method = 'fft_2D'; 
% remove phase ramp from data to using self consistency  
stack_object = tomo.phase_ramp_removal_tomo(stack_object,selected_ROI, theta, Npix_align*1.2, total_shift, par, 'binning', binning, 'Niter', Niter, ...
    'positivity', false, 'auto_weighting', true,  'sino_weights', weight_sino);

if ~par.online_tomo || debug()
    % show the sorted projection, check if all are ok. If not try to run
    % alignment again. Maybe change vertical range or high_pass filter 
    tomo.show_projections(stack_object, theta, par, 'fnct', @(x)angle(x(object_ROI{:},:)),  'plot_residua', false)
end

%%%%%%%%%%%%%%%%%%%%%%%%%%%
% End of phase refinement 
%%%%%%%%%%%%%%%%%%%%%%%%%%%


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%% Save aligned complex projections for external processing 
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%% Edit this section %%%
%%%%%%%%%%%%%%%%%%%%%%%%%
save_external = false; 
%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%
if save_external && ~par.online_tomo && ~debug()
    utils.savefast_safe([par.output_folder,'/stack_object_external'], 'stack_object', 'theta', 'par', 'total_shift', 'object_ROI')
end

%% (EXPERIMENTAL FEATURE) Find optimal propagation (numerical refocusing), try to minimize amplitude in the complex projections 
%%%%%%%%%%%%%%%%%%%%%%%%%
%%% Edit this section %%%
%%%%%%%%%%%%%%%%%%%%%%%%%
refocus_projections = false;
refocusing_range = linspace(-8,8,10)*1e-6;               % scanning range 
object_ROI_prop = object_ROI_xcorr; % so that it doesn't contain noise from the corners
%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%
     
if refocus_projections 
    optimal_propagation = projection_propagation_optimization( stack_object, theta, refocusing_range, object_ROI_prop, par); 
    if strcmpi(input('Apply estimated propagation ? [y/N]\n', 's'), 'y')
        stack_object = tomo.block_fun(@utils.prop_free_nf, stack_object, par.lambda, optimal_propagation(:), par.pixel_size,  struct('GPU_list',par.GPU_list)); 
    end
end

%% Full tomogram
%%%%%%%%%%%%%%%%%%%%%%%%%
%%% Edit this section %%%
%%%%%%%%%%%%%%%%%%%%%%%%%
par.usecircle = true;               % Use circle to mask out the corners of the tomogram
par.filter_type = 'ram-lak';        % FBP filter (ram-lak, hamming, hann....)
par.freq_scale = 1;                 % Frequency cutoff
rec_ind = find(par.valid_angles);   % use only some angles 
apodize = 0;                        % axial apodization 
radial_smooth_apodize = 20;         % smooth the apodizing function 
%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%

circulo = [];  % apodization function 
if par.cache_stack_object && ~exist('stack_object', 'var')
    stack_object = load_stored_object('stack_object'); 
end

% find optimal regions from stack_object
min_proj = tomo.block_fun(@abs, stack_object, struct('use_GPU', false, 'reduce_fun', @min)); 
max_proj = tomo.block_fun(@abs, stack_object, struct('use_GPU', false, 'reduce_fun', @max)); 

if par.is_laminography
    % for laminography select the region that is covered by all projections
    reconstruct_ROI = get_ROI(min_proj>0.1*max(min_proj(:)), 0, 32);
else  % othrewise keep the selected ROI used for alignment (change if needed)
    reconstruct_ROI = selected_ROI;  
end
    

utils.verbose(-1,'Generating sinograms')
clear sinogram 
 % 2D phase unwrapping 
weight_sino = tomo.block_fun(@lamino.estimate_reliability_region,stack_object, par.asize, 16);


% unwrap the complex phase 
preprocess_fun = @(x)(utils.imshear_fft(x, -par.skewness_angle,1));  % shear has to be applied before ASTRA reconstruction (block splitting does not work well with shear)
sinogram = -tomo.unwrap2D_fft2_split(stack_object,par.air_gap,0,weight_sino, par.GPU_list, reconstruct_ROI,preprocess_fun);


if par.cache_stack_object
    clear stack_object  % empty memory before trying to make reconstruction
end

Nlayers_rec = length(reconstruct_ROI{1}); 
Nw_rec = length(reconstruct_ROI{2}); 

if par.is_laminography
    %% in case of laminography 
    Npix = ceil(0.5/cosd(par.lamino_angle)*Nw_rec);  % for pillar it can be the same as Nw_rec
    Npix = ceil([Npix, Npix, sample_thickness /par.pixel_size]/32)*32; 
else
    %% limited angle tomo 
    Npix = ceil(1.2*Nw_rec/sqrt(2)/32)*32;  % for pillar it can be the same as Nw_rec
    Npix = [Npix/4, Npix, Nlayers_rec];
end


% Get two reconstructions for FSC   %%%%%%%%%%%%%%
[~,ind_sort] = sort(theta); 
ind_sort = ind_sort(ismember(ind_sort, rec_ind));
ind_rec = {ind_sort(1:2:end), ind_sort(2:2:end)}; 


CoR_offset = param.center_of_rotation(2) - ...
            (length(reconstruct_ROI{2})/2 -0.5 + ... 
                 reconstruct_ROI{2}(1)-1);  

% remove artefacts around edges of tomogram 
if par.usecircle
    [~,circulo] = utils.apply_3D_apodization(ones(Npix(1:2)), apodize, 0, radial_smooth_apodize); 
end

 
% Npix = ceil(0.5/cosd(par.lamino_angle)*Nw_rec);  % for pillar it can be the same as Nw_rec
% Npix = ceil([Npix, Npix, sample_thickness /par.pixel_size]/32)*32; 

[cfg, vectors] = astra.ASTRA_initialize(Npix,[Nlayers_rec,Nw_rec],theta,par.lamino_angle,par.tilt_angle,1, [Nlayers_rec,Nw_rec]/2 + [0,CoR_offset]); 
% find optimal split of the dataset for given GPU 
split = astra.ASTRA_find_optimal_split(cfg, length(par.GPU_list), 2, 'back');

for ii = 1:2
    % new FBP code 
    tomograms{ii} = tomo.FBP(sinogram, cfg, vectors, split,'valid_angles',ind_rec{ii},...
        'GPU', par.GPU_list,'filter',par.filter_type, 'filter_value',par.freq_scale,...
        'use_derivative', false, 'mask', circulo, 'padding', 'symmetric');
end
if par.save_memory
    clear sinogram  % clear the sinogram as soon as possible 
end

%% 3D FSC 
%%%%%%%%%%%%%%%%%%%%%%%%%
%%% Edit this section %%%
%%%%%%%%%%%%%%%%%%%%%%%%%
show_figures = true;
rad_apod = Npix(1)/5;           % Radial apodization
axial_apod = 20;                % Axial apodization
radial_smooth = Npix(1)/10;     % Smoothness range of the apodization function 
SNRt = 0.2071; %1/2 bit         % Threshold curve for FSC
thickring = 5;                  % use thickring for smoothing 
FSC_vertical_range =10:Npix(3)-10;  % Choose region for FSC
%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%

% call "close(45)" if you want to plot FSC to clean window (without overlay with previous FSC curve)
[resolution FSC T freq n FSC_stats, fsc_path] = tomo.get_FSC_from_subtomos(tomograms, FSC_vertical_range, rad_apod, radial_smooth, axial_apod,SNRt,thickring,par);
utils.savefast_safe([fsc_path, '.mat'], 'FSC','T','freq','n','FSC_stats', 'rad_apod','axial_apod','circulo','theta', 'total_shift', 'par', par.force_overwrite || par.online_tomo);





%% Caclulate delta tomogram 

% get full reconstruction (for FBP is sum already final tomogram)
% calculate complex refractive index 
par.rec_delta_info = ['FBP_',par.filter_type '_freqscl_' sprintf('%0.2f',par.freq_scale)]; 
tomogram_delta = ((tomograms{1} + tomograms{2})/2)*par.factor;
% clear the tomograms for FSC from RAM 
clear tomograms  

if ~debug()
    figure(131)
    plotting.imagesc_tomo(tomogram_delta); 
    suptitle('Tomogram delta - FBP reconstruction')
end



%%  SART  solver - helps mainly for sparse sample (with a lot of air gaps) and in case of angularly undersampled tomograms 
%%   - solver uses positivity constraint 
%%%%%%%%%%%%%%%%%%%%%%%%%
%%% Edit this section %%%
%%%%%%%%%%%%%%%%%%%%%%%%%
par.Niter_SART = 0 + (debug>0 && debug<3);     % number of SART iteration 
SART_block_size = 100;                         % size of blocks solved in parallel
par.usecircle = true;                          % Use circle to mask out the corners of the tomogram
relax = 0 ;                                    % SART relaxation, 0 = no relaxation
%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%


if par.Niter_SART > 0
    if par.usecircle
        [~,circulo] = utils.apply_3D_apodization(ones(Npix), apodize, 0, radial_smooth_apodize);
    else
        circulo = 1; 
    end

    utils.verbose(-1,'SART preparation', ii, par.Niter_SART)
    [Ny_sino,Nx_sino,~] = size(sinogram);
    [cfg, vectors] = astra.ASTRA_initialize(Npix,[Nlayers_rec,Nw_rec],theta,par.lamino_angle,par.tilt_angle,1, [Nlayers_rec,Nw_rec]/2 + [0,CoR_offset]); 
    
    split = astra.ASTRA_find_optimal_split(cfg, length(par.GPU_list), 1, 'both');
    [cache_SART, cfg] = tomo.SART_prepare(cfg, vectors, SART_block_size, split);

    tomogram_SART = tomogram_delta/par.factor;
    clear err_sart 
    for ii = 1:par.Niter_SART
        utils.verbose(-1,'SART iter %i/%i', ii, par.Niter_SART)
        [tomogram_SART,err_sart(ii,:)] = tomo.SART(tomogram_SART, sinogram, cfg, vectors, cache_SART, split, ...
            'relax',relax, 'constraint', @(x)(abs(x) .* (0.9+0.1*circulo))) ; 
        plotting.smart_figure(1111)
        ax(1)=subplot(2,2,1);
        plotting.imagesc3D(tomogram_delta, 'init_frame', size(tomogram_delta,3)/2); axis off image; colormap bone; 
        title('Original FBP reconstruction')
        ax(2)=subplot(2,2,2);
        plotting.imagesc3D(tomogram_SART, 'init_frame', size(tomogram_delta,3)/2); axis off image; colormap bone;  
        title('Current SART reconstruction')
        subplot(2,1,2)
        plot(1:ii,err_sart)
        hold all
        plot(1:ii,median(err_sart,2),'k--', 'Linewidth',4)
        hold off SART
        xlabel('Iteration #'); ylabel('Error'); set(gca, 'xscale', 'log'); set(gca, 'yscale', 'log')
        grid on  
        title('Evolution of projection-space error')
        linkaxes(ax, 'xy')
        plotting.suptitle('SART reconstruction')
        drawnow
    end
    
    figure
    plotting.imagesc_tomo(tomogram_SART); 
    plotting.suptitle('Tomogram delta - FBP reconstruction')
    if  par.online_tomo || debug() || ~strcmpi(input('Should the SART reconstruction be saved to tomogram_delta ? [Y/n]'), 'n')
        % store SART into tomogram_delta
        tomogram_delta = tomogram_SART * par.factor; 
        par.rec_delta_info = sprintf('SART_Niter=%i', par.Niter_SART); 
    end
     clear tomogram_SART cache_SART
end

clear sinogram 







%% ROTATE THE TOMOGRAM 
% try to find a most significant direction in the sample and align along
% it, useful for chips :)
% 
% [rot_angle(1)] = utils.find_img_rotation_2D(utils.Garray(squeeze(tomogram_delta(end/2,:,:))));
% [rot_angle(2)] = utils.find_img_rotation_2D(utils.Garray(squeeze(tomogram_delta(:,end/2,:))));
% [rot_angle(3)] = utils.find_img_rotation_2D(utils.Garray(mean(tomogram_delta,3)));
% 
% utils.verbose(-1,'Rotating')
% tomogram_delta = permute(tomo.block_fun(@utils.imrotate_ax_fft,permute(tomogram_delta,[3,2,1]),-rot_angle(1)),[3,2,1]);
% tomogram_delta = permute(tomo.block_fun(@utils.imrotate_ax_fft,permute(tomogram_delta,[1,3,2]), rot_angle(2)),[1,3,2]);
% tomogram_delta = tomo.block_fun(@utils.imrotate_ax_fft,tomogram_delta, rot_angle(3));
% utils.verbose(-1,'Done')
% 
% % % show quick preview 
% figure; plotting.imagesc_tomo(tomogram_delta)



%% Display reconstructed volume by slices
%%%%%%%%%%%%%%%%%%%%%%%%%
%%% Edit this section %%%
%%%%%%%%%%%%%%%%%%%%%%%%%
par.displayaxis = 3;            % 1 coronal, 2 sagital, 3 axial
par.displayslice = [];         % Many slices supported e.g [100:200]. If left empty the center slice is shown, 
par.animatedslices = false;     % All slices are shown ignoring those in displayslice
par.average_slices = false;    % If true, it ignores animatedslices
% Display options
par.tomobaraxis = 'auto';  % [-5,0.2];       % = 'auto', 'auto_per_frame'  or e.g.  = [-1 1]. For auto it picks max and min from whole tomogram
par.scale = 'edensity';            % = 'phase', 'delta' [index] or 'edensity' [electron/A^3]
par.colormapchoice = 'bone';    % Choose the colormap
par.realaxis = true;            % = true to show result with real axes units, = false for pixels
par.reverse_contrast = true;    % Reverse grayscale
% Scale bar options (only shown if realaxis = true)
par.bar_length= 2e-6;           % Lenght of scalebar (only used if realaxis = true)
par.bar_height= 0.2e-6;           % Height of scalebar (only used if realaxis = true)
par.bar_start_point=[0.8 0.2]*1e-6;    % (x, y) starting point of the scale bar (from corner) in meters
par.bar_color = 'k';            % Color of scalebar
% Other
par.windowautopos = true;      % If true the window position and size is changed, can be useful as false when ready to render a movie
par.makemovie = false;          % = true for rendering an AVI
par.writesnapshots = false;      % Makes jpeg and eps snaphots. It overwrites snapshots. If many slices are chosen it writes only the last one.
par.pausetime = 0.1;            %   use time for animation or multiple slices
%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%

% Coronal, first index
% Sagital, second index
% Axial,   third index

tomo.show_tomogram_cuts(tomogram_delta, par.scanstomo, par, par.rec_delta_info)

%% save images for Database and online tomo 
if ~isempty(par.tomo_id) && ~debug()
    upload_image_to_OMNY_database(tomogram_delta, par);
end

%% Save Tomogram (delta)
%%%%%%%%%%%%%%%%%%%%%%%%%
%%% Edit this section %%%
%%%%%%%%%%%%%%%%%%%%%%%%%
savedata = true;
%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%
tic

% Note saving tomogram in delta (index of refraction)
% To get phase tomogram   = tomogram_delta/par.factor
% To get electron density tomogram in [e/A^3]   = tomogram_delta*par.factor_edensity
% If you saved before you may clear 
if savedata && ~debug()
    tomo.save_tomogram(tomogram_delta, par, 'delta', circulo, theta, par.rec_delta_info)
end
toc


%% Save Tiff files for 3D visualization with external program
%%%%%%%%%%%%%%%%%%%%%%%%%
%%% Edit this section %%%
%%%%%%%%%%%%%%%%%%%%%%%%%
par.save_as_stack = false; 
par.tiff_compression = 'none';
par.tiff_subfolder_name = ['TIFF_delta_' par.rec_delta_info];
par.name_prefix = 'tomo_delta';
%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%
if ~debug()
    tomo.save_as_tiff(tomogram_delta, par, par.rec_delta_info)
end

%% REGULARIZE LAMINOGRAPHY 



%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%% %%%%%%%%%   FILL IN THE MISSING INFORMATION IN THE MISSING CONE  - TESTED ONLY ON CHIPS %%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

%%%%%%%%%%%%%%%%%%%%%%%%%
%%% Edit this section %%%
%%%%%%%%%%%%%%%%%%%%%%%%%
fill_missing_cone = false || debug();      % if true, run this section and fill the missing cone 
delta_background = 1.24e-05;    % the lowest expected delta in the sample (ie silicon oxide in the chip)
delta_maximal = 4.3e-5;         % the maximal extected delta -> Cu in the chip 
mask_relax = 0.05;              % strength of pushing the masked values to zeros -> enforce low values in close to empty regions 
max_scale = 16; % process the reconstruction using multiscale approach, start at 0.5^max_scale
Niter = 100; % number of iterations in each step for filling the missing cone 
extra_string_tiff = '';       % extra string for tiff folder and saved .mat
%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%

if debug() == 2  % synthetic data tests 
   % !! DONT CHANGE!! , default values only for automatic tests 
   delta_background = 0;       % the lowest expected delta in the sample
   delta_maximal = 0.03;       % the maximal extected delta 
elseif debug()  == 3  % Lamni chip dataset 
   delta_background = 1.2e-5;    % delta SiO2
   delta_maximal = 4.26e-5;      % delta Cu
end

if fill_missing_cone

rec = tomogram_delta; 
rec = rec - median(rec(:)); 
rec0 = rec; 

Npix = size(rec);


% weakly suppress nonzero values in empty regions (along vertical axis )
mask_vert = math.mean2(abs(rec0)); 
mask_vert = (1-mask_relax+mask_relax*mask_vert / max(mask_vert)); 



Npix_full = size(rec);
max_block_size = [512,512]; 
border_size = [32,32]; 
block_size = [512,512];
utils.verbose(-1,'Filling missing cone')
for scale = 2.^(log2(max_scale):-1:0)
    utils.verbose(-1,'Running scale %i', scale)
    if scale > 1
        rec_small = utils.interpolateFT_3D(rec, ceil(Npix_full/scale));
        mask_vert_small = utils.interpolateFT_3D(mask_vert, [1,1,ceil(Npix_full(3)/scale)]); 
    else
        rec_small = rec; 
        mask_vert_small = mask_vert; 
    end
    low_freq_protection = scale < max_scale; 

    blkfun_cfg = struct('Nblocks',ceil(size(rec_small,3)/512), 'GPU_list', par.GPU_list, 'verbose',0); 

    % enforce "material constraint", ie force reconstruction to be positive
    % and smaller then provided threshold (ie delta_material - delta_background)
    %  !! due to limitations of GPU code, it has to be hardcoded value in
    %  this function 
    
    % apply the constraints on the missing cone region in FFT space, solve
    % it blockwise if the object is too large 
    tic
    blockprocess = @(x)tomo.block_fun(@lamino.apply_lamino_constraints, x.data, mask_vert_small, par.lamino_angle, low_freq_protection, delta_maximal, delta_background, Niter, blkfun_cfg); 
    rec_regularized = blockproc(rec_small,block_size-2*border_size, blockprocess, 'BorderSize', border_size, 'DisplayWaitbar', true, 'PadMethod',  'symmetric');
    toc
    
    % update the reconstruction by an upscaled difference before and after
    % the missing cone filling 
    rec = rec + utils.interpolateFT_3D(rec_regularized - rec_small, Npix_full);
    clear rec_regularized rec_small 
    
    % plot progress 
    plotting.smart_figure(201)
    ax(1) = subplot(2,1,1);
    imagesc(rot90(squeeze(rec0(:,end/2,:))), [delta_background, delta_maximal]);
    axis off image 
    colormap bone 
    title('Lamino reconstruction')
    ax(2) = subplot(2,1,2);
    imagesc(rot90(squeeze(rec(:,end/2,:))), [delta_background, delta_maximal]);
    axis off image 
    title('Lamino refined reconstruction')
    drawnow 

end


% compare quality of the missing cone filling in the fourier space 
plotting.smart_figure(202)
ax(1)=subplot(1,2,1); 
frec = gather(fftshift((fftn(rec0))));
imagesc(log(1e-3+abs(rot90(squeeze(frec(:,end/2,:))))))
axis off square 
colormap bone 
title('log FFT of original reconstruction')
ax(2)=subplot(1,2,2); 
frec = gather(fftshift((fftn(fftshift(rec)))));
imagesc(log(1e-3+abs(rot90(squeeze(frec(:,end/2,:))))))
axis off square 
colormap bone 
title('log FFT of refined reconstruction')
drawnow 

    if ~debug()
    %%%%%%%%% save result under a different name 
    tomo.save_tomogram(rec, par, 'delta', '', theta, ['positivity+TV_regularized' extra_string_tiff])

    %%%%%%%% save refined TIFF files 
    par.save_as_stack = false; 
    par.tiff_compression = 'none';
    par.tiff_subfolder_name = ['TIFF_delta_regularized_' par.filter_type '_freqscl_' sprintf('%0.2f',par.freq_scale) extra_string_tiff];
    par.name_prefix = 'tomo_delta';
    %%%%%%%%%%%%%%%%%%%%%%%%%
    %%%%%%%%%%%%%%%%%%%%%%%%%
    tomo.save_as_tiff(rec, par,extra_string_tiff)
end
clear rec rec0 frec 
    
end

if ~debug()
%% Determine calibration for sphere for auto-alignment tomography
par.surface_calib_file = [base_path 'matlab/tomo/position_calibration_2018_test.mat'];

tomo.give_calibration(obj_interf_pos_x, obj_interf_pos_y , -total_shift(:,[2,1])', 0, theta, par.scanstomo, par);
end


%% Dose estimation
%%%%%%%%%%%%%%%%%%%%%%%%%
%%% Edit this section %%%
%%%%%%%%%%%%%%%%%%%%%%%%
par.mu = 1/(451*1e-6); % 1/attenuation_length   in 1/m   (for CH2 @6.2keV)
                     % 1/(152.7*1e-6) for zeolite Na2Al2Si3O102H4O with 2 g/cm3 density at 6.2 keV
par.rho = 1000;          % Density in kg/m^3
par.setup_transmission = 0.98; % Intensity transmission of sample 
                        % (e.g. air path after the sample, windows, He, detector efficiency)
                        % 0.943 for 700 cm He gas at 760 Torr and 295 K @ 6.2 keV
                        % 0.780 for 10 cm air at 760 Torr and 295 K @ 6.2 keV
                        % 0.976 for 13 micron Kapton (polymide) with 1.43
                        % g/cm3 @ 6.2 keV
                        % 0.841 for 7 micron muskovite mica
                        % (KAl3Si3O11.8H1.8F0.2) with 2.76 g/cm3 @ 6.2 keV
                        % 0.914 for 5 cm of air at 6.2 keV 750 Torr 295 K
                        % 0.55 for 300 micron of mylar C10H8O4 with density 1.38 g/cm3 at 6.2 keV
par.overhead = 0.0;         % Extra dose during movement overhead, only applicable 
                        %      if shutter is not closed between exposures
                        %      (meastime/exptime - 1)
if ~debug()
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%  
scan_0 = par.scanstomo(1);   % Choose scan number to be used as an example 
ptycho_filename = find_projection_files_names(par, scan_0);
ptycho_recon = io.load_ptycho_recons(ptycho_filename);
[ ptycho_data.data, ptycho_data.fmask] = io.load_prepared_data( ptycho_filename, true,[],false); 



utils.dose_calc(ptycho_recon,ptycho_data,par);

% Clear memory 
clear ptycho_recon ptycho_data 
end

if par.save_memory
   clear tomogram_delta
end

%% Tomography with amplitude

if par.cache_stack_object && ~exist('stack_object', 'var')
    stack_object = load_stored_object('stack_object'); 
end


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%% Show movie with aligned images
%%%%%%%%%%%%%%%%%%%%%%%%%
%%% Edit this section %%%
%%%%%%%%%%%%%%%%%%%%%%%%%
par.baraxis = 'auto';       % = 'auto'  or   = [-1 1]
par.windowautopos = true;  % automatic placemement of the plot
par.showsorted = true;      % sort the projections by angles 
%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%
tomo.show_projections(stack_object, theta, par, 'fnct',  @(x)abs(x(object_ROI{:})))



%% Get attenuation sinogram (use the same ROI as for phase reconstructions)

% get attenuation 
preprocess_fun = @(x)(utils.imshear_fft(x, -par.skewness_angle,1));  % shear has to be applied before ASTRA reconstruction (block splitting does not work well with shear)
sino_amp = tomo.block_fun(@(x)(-log(abs(preprocess_fun(x)))), stack_object,struct('use_GPU',false,'use_fp16',false,'ROI',{reconstruct_ROI}));


if par.cache_stack_object 
    clear stack_object
end

%% Full tomogram with amplitude
%%%%%%%%%%%%%%%%%%%%%%%%%
%%% Edit this section %%%
%%%%%%%%%%%%%%%%%%%%%%%%%
usecircle = true;            % Use circle to mask out the corners of the tomogram
filter_type_amp = 'ram-lak'; % FBP filter (ram-lak, hamming, ....)
freq_scale_amp = 1;       % FBP filter value 
apodize = 0;                 % number of pixels around edges to be removed 
% NOTE: vertical range is identical to the phase reconstruction range 
%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%


[cfg, vectors] = astra.ASTRA_initialize(Npix,[Nlayers_rec,Nw_rec],theta,par.lamino_angle,par.tilt_angle,1, [Nlayers_rec,Nw_rec]/2 + [0,CoR_offset]); 
% find optimal split of the dataset for given GPU 
split = astra.ASTRA_find_optimal_split(cfg, length(par.GPU_list), 1, 'back');

if usecircle
    [~,circulo] = utils.apply_3D_apodization(ones(Npix(1:2)), apodize, 0, radial_smooth_apodize); 
end
tomogram_beta = par.factor*tomo.FBP(sino_amp, cfg, vectors, split,...
    'GPU', par.GPU_list,'filter',par.filter_type, 'filter_value',par.freq_scale,...
    'mask', circulo, 'padding', 'symmetric');

par.rec_beta_info = ['FBP_',par.filter_type '_freqscl_' sprintf('%0.2f',par.freq_scale)]; 

% if provided, rotate tomogram to get features parallel with Z-plane
if exist('rot_angle', 'var')
    utils.verbose(-1,'Rotating')
    tomogram_beta = permute(tomo.block_fun(@utils.imrotate_ax_fft,permute(tomogram_beta,[3,2,1]),-rot_angle(1)),[3,2,1]);
    tomogram_beta = permute(tomo.block_fun(@utils.imrotate_ax_fft,permute(tomogram_beta,[1,3,2]), rot_angle(2)),[1,3,2]);
    tomogram_beta = tomo.block_fun(@utils.imrotate_ax_fft,tomogram_beta, rot_angle(3));
    utils.verbose(-1,'Done')
end
if par.save_memory
    clear sino_amp
end


%% Display reconstructed volume by slices
%%%%%%%%%%%%%%%%%%%%%%%%%
%%% Edit this section %%%
%%%%%%%%%%%%%%%%%%%%%%%%%
par.displayaxis = 3;            % 1 coronal, 2 sagital, 3 axial
par.displayslice = [];         % Many slices supported e.g [100:200]. If left empty the center slice is shown, 
par.animatedslices = false;     % All slices are shown ignoring those in displayslice
par.average_slices = false;    % If true, it ignores animatedslices
% Display options
par.tomobaraxis = 'auto';       % = 'auto'  or e.g.  = [-1 1]. For auto it picks max and min from whole tomogram
par.scale = 'beta';            % = 'phase', 'delta' [index] or 'edensity' [electron/A^3]
par.colormapchoice = 'bone';    % Choose the colormap
par.realaxis = true;            % = true to show result with real axes units, = false for pixels
par.reverse_contrast = false;    % Reverse grayscale
% Scale bar options (only shown if realaxis = true)
par.bar_length= 2e-6;           % Lenght of scalebar (only used if realaxis = true)
par.bar_height= 0.2e-6;           % Height of scalebar (only used if realaxis = true)
par.bar_start_point=[0.8 0.2]*1e-6;    % (x, y) starting point of the scale bar (from corner) in meters
par.bar_color = 'w';            % Color of scalebar
 % Other
par.windowautopos = true;      % If true the window position and size is changed, can be useful as false when ready to render a movie
par.makemovie = false;          % = true for rendering an AVI
par.writesnapshots = false;      % Makes jpeg and eps snaphots. It overwrites snapshots. If many slices are chosen it writes only the last one.
par.pausetime = 0.1;            % Pause time for animation or multiple slices
%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%

% Coronal, first index
% Sagital, second index
% Axial,   third index

tomo.show_tomogram_cuts(tomogram_beta, par.scanstomo, par, par.rec_beta_info)


%% Save Tomogram (beta)
%%%%%%%%%%%%%%%%%%%%%%%%%
%%% Edit this section %%%
%%%%%%%%%%%%%%%%%%%%%%%%%
savedata = true;
%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%
% Note saving tomogram in delta (index of refraction)
% To get phase tomogram   = -tomgram_delta/factor
% To get electron density tomogram in [e/A^3]   = tomogram_delta*factor_edensity
% If you saved before you may clear 
if savedata && ~debug()
    tomo.save_tomogram(tomogram_beta, par, 'beta', circulo,  theta, par.rec_beta_info)
end


%% Save Tiff files for 3D visualization with external program
%%%%%%%%%%%%%%%%%%%%%%%%%
%%% Edit this section %%%
%%%%%%%%%%%%%%%%%%%%%%%%%
par.save_as_stack = false; 
par.tiff_compression = 'none';
par.tiff_subfolder_name = ['TIFF_beta_' par.rec_beta_info];
par.name_prefix = 'tomo_beta';
%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%
if ~debug()
    tomo.save_as_tiff(tomogram_beta, par, par.rec_beta_info)
end


utils.verbose(-1,'=========== LAMINOGRAPHY ALIGNMENT AND RECONSTRUCTION FINISHED ==============')
reset(gpuDevice)  % make sure that the GPU memory is empty 


%*-----------------------------------------------------------------------*
%|                                                                       |
%|  Except where otherwise noted, this work is licensed under a          |
%|  Creative Commons Attribution-NonCommercial-ShareAlike 4.0            |
%|  International (CC BY-NC-SA 4.0) license.                             |
%|                                                                       |
%|  Copyright (c) 2017 by Paul Scherrer Institute (http://www.psi.ch)    |
%|                                                                       |
%|       Author: CXS group, PSI                                          |
%*-----------------------------------------------------------------------*
% You may use this code with the following provisions:
%
% If the code is fully or partially redistributed, or rewritten in another
%   computing language this notice should be included in the redistribution.
%
% If this code, or subfunctions or parts of it, is used for research in a 
%   publication or if it is fully or partially rewritten for another 
%   computing language the authors and institution should be acknowledged 
%   in written form in the publication: “Data processing was carried out 
%   using the “cSAXS matlab package” developed by the CXS group,
%   Paul Scherrer Institut, Switzerland.” 
%   Variations on the latter text can be incorporated upon discussion with 
%   the CXS group if needed to more specifically reflect the use of the package 
%   for the published work.
%
% A publication that focuses on describing features, or parameters, that
%    are already existing in the code should be first discussed with the
%    authors.
%   
% This code and subroutines are part of a continuous development, they 
%    are provided “as they are” without guarantees or liability on part
%    of PSI or the authors. It is the user responsibility to ensure its 
%    proper use and the correctness of the results.
