%%%%%%%%%%%%%% README %%%%%%%%%%%%%%%%%%%%%%%%
% Template for nonrigid tomography reconstruction and 
%   estimation of time-evolving deformation vector fields
% HOw TO USE: 
%   1) make sure that cSAXS_matlab_base package is unpacked in
%   ./cSAXS_matlab_base/
%   2) set use_artif_data = true to run phantom-based reconstruction or 
%      set use_artif_data = false to reconstruct the provided example of
%      real data tored in "cache_Beetle_deform.mat"
%   3) run template. Instructions: First perform alignment with standart tomography code with the setting save_external=true (generating the file stack_object_external.mat). 
%   Load that file into this tamplate by specifying path_to_projections and projection_filename. Run template.
%
%   Requirements:
%      Tested with Matlab 2018a CUDA 9.0 with parallel toolbox on Linux
%      other configurations may need recompilation of MEX binaries 
%      Run following commands in the folder 
%      ./cSAXS_matlab_base/+astra
%               (Linux, GCC 4.8.5)   mexcuda -outdir private  ASTRA_GPU_wrapper/ASTRA_GPU_wrapper.cu ASTRA_GPU_wrapper/util3d.cu ASTRA_GPU_wrapper/par3d_fp.cu ASTRA_GPU_wrapper/par3d_bp.cu
%               (Windows)            mexcuda -outdir private  ASTRA_GPU_wrapper\ASTRA_GPU_wrapper.cu ASTRA_GPU_wrapper\util3d.cu ASTRA_GPU_wrapper\par3d_fp.cu ASTRA_GPU_wrapper\par3d_bp.cu
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%% template for nonrigid tomography %%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%





%% Add paths
base_path='../../';
addpath('utils')
addpath('tests')
addpath(find_base_package())

close all 


%%%%%%%%%%%%%%%%%%%%%%%%%%
%% general setting 
%%%%%%%%%%%%%%%%%%%%%%%%%%
GPU = 1;                            % ID of the used GPU 
use_artif_data = false || debug();             % automatically assume use_artif_data = true if called from externally;
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

if ~use_artif_data
    %%%%%%%%%%%%%%%%%%%%%%%%%%
    %% real data LOADING   
    %%%%%%%%%%%%%%%%%%%%%%%%%%
    path_to_projections = '../../analysis_tomo/tomo__id_179_S01492_to_S02201_800x800_b0_run_1_recons_/'; 
    projection_filename = 'stack_object_external.mat'; 
    sample_name = 'beetle'; 

    binning = 4;                        % find the nonrigid deformation on a binned dataset 
    Nblocks = 8;                        % number of blocks (subtomograms) 
    par.rigid_correction = true;      % correct also shifts of the projections 
    par.nonrigid_correction = true;   % find optimal DVF 

    %%%%%%%%%%%%%%%%%%%%%%%%%%%%%
    %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
else
    %%%%%%%%%%%%%%%%%%%%%%%%%%
    %% artificial data GENERATION 
    %%%%%%%%%%%%%%%%%%%%%%%%%%
    binning  = 1;
    Npix_0 = 200;  % volume dimension 
    Nlayers = 128;
    Nblocks = 5; 
    Nangles = 400;
    sample_name = 'artificial';
    DVF_period = 10; 
    DVF_amplitude = 10; 
    par.rigid_correction = false;      % correct also shifts of the projections 
    par.nonrigid_correction = true;   % find optimal DVF 
    %%%%%%%%%%%%%%%%%%%%%%%%%%
    smooth = 10;                % smoothness of the produced DVF
    %%%%%%%%%%%%%%%%%%%%%%%%%%%%%
    %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
end


%%%%%%%%%%%%%%%%%%%%%%%%%%
%% same basic optimization parameters 
%%%%%%%%%%%%%%%%%%%%%%%%%%

par.Niter_registration = 20;      % number of iteration for the optimal DVF search 
par.max_smooth = 80;              % initial smoothing of the DVF 
par.min_smooth = 30;              % final smoothing 
par.downsample_DVF = 4;           % how many pixels in reconstruction is described by one pixel of DVF
par.regular = 1e-1;               % prevent amplification of noise in DVF inside of empty regions 
par.use_mask = true;              % apply mask on SART reconstructions 
par.mask_threshold  = 0.1;        % threshold * mean(tomo) is used to guess reconstruction mask 
par.mask_dilate = 0.1;            % dilate mask estimate by mask_dilate * Npix
par.relax_pos_corr = 0.8;         % move only relax_pos_corr - times the distance compared to the optimal -> avoid oscilations 
par.regularize_deform_evol = 0.02;   % temporal smoothness of the deformation array 
par.method = 'FBP';               % default reconstruction method during the DVF search: FBS, SART, CLGS
par.Niter_SART= 5;                % number of inner iterations for SART method
par.plot_every = 20;              % results are plotting ever x second 
par.GPU_list = GPU;               % list of the used GPUs

%% other default settings 
par.air_gap = [20,20]; % region of air around sample for phase unwrapping 
par.output_folder = ''; 
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%



%% initialize %%%%%%%%%%%
addpath('utils')
addpath('tests')

import nonrigid.*
import plotting.*

% reset(gpuDevice(GPU))
gpu = gpuDevice(GPU); 
if gpu.Index ~= GPU
    disp('reset GPU')
    gpu = gpuDevice(GPU);
end
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

if ~use_artif_data
    % load real data, use tomo_template to save prealigned and preprocessed projections 
    [dphase_0_full,~, theta, par, object_0_full] = ...
            nonrigid.prepare_real_deform_data(path_to_projections,projection_filename,sample_name, par); 
else
    % only pretent to load data and provide an artificial dataset 
    [dphase_0_full, deform_tensor_orig, theta, par, rec_ideal, volData_orig, ] = ...
        nonrigid.prepare_artificial_deform_data(Nangles, Npix_0, Nlayers, Nblocks, smooth, binning, DVF_period, DVF_amplitude, par); 

end



%% apply "binning"
[Nlayers,Nw,Nangles]=size(dphase_0_full);
dphase_0 = real(utils.interpolateFT(dphase_0_full,ceil([Nlayers,Nw]/binning)));
dphase_0 = utils.smooth_edges(dphase_0);
par.air_gap = par.air_gap/binning; 


%% prepare other inputs 
[Nlayers,Nw,Ntheta]=size(dphase_0);
Npix = ceil(Nw/16)*16; % make is easily splitable size 

%% modify center of rotation to account for the applied binning of the projections (keep subpixel accuracy)
rotation_center = [Nlayers, Nw]/2; 
CoR_offset = - 0.5*(1-1/binning);        % apply coordinates correction for the "binning" by FFT
rotation_center(2) = rotation_center(2) + CoR_offset ;   % important for binning => account for additional shift of the center of rotation after binning, for binning == 1 the correction is zero
   
Bsize = ceil(Ntheta/Nblocks); 
Nps = ceil([Npix, Npix, Nlayers]/par.downsample_DVF);

%% get geometry vectors & configuration structure for ASTRA matlab wrapper 
[cfg, vectors] = ...
    astra.ASTRA_initialize([Npix, Npix,Nlayers],[Nlayers,Nw],theta,90, 0, 1, rotation_center);
split = astra.ASTRA_find_optimal_split(cfg);






% get initial reconstruction (conventional rigid FBP method from phase derivative data)
rec_0 = gather(tomo.FBP(dphase_0, cfg, vectors,'use_derivative', true)) ;


% get weights for reconstruction 
W_sino = (single(tukeywin(Nlayers, 0.2)  .*  tukeywin(Nw, 0.2)'));  % avoid edge issues 
[~, W_rec] = nonrigid.get_mask(rec_0, par.mask_threshold, par.mask_dilate * Npix); 

if par.use_mask
    [~,rec_mask] = utils.apply_3D_apodization(rec_0, 0); 
else
    rec_mask = []; 
end
    
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%% initialization %%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
shift_3D_total = cell(Nblocks,1);
for ll = 1:Nblocks
    for kk  = 1:3
         shift_3D_total{ll}{kk} = gpuArray.zeros(Nps, 'single');
    end
end
% store accumulated shift of the ASTRA geometry 
total_error = zeros(par.Niter_registration, Ntheta); 
total_geom_shift = zeros(par.Niter_registration, Ntheta, 2); 
smooth = par.max_smooth;

clear err TV_score rec_center_0
t_plot = tic; 
      
rec_avg = rec_0 ;
dphase = dphase_0; 

if strcmp('SART', par.method)
    [SART_cache_all, cfg] = tomo.SART_prepare(cfg, vectors, Bsize, split);
end



disp('Recoverging 3D deformation ... ')


%%
for iter = 1:par.Niter_registration
    % generate deformation tensors from the shift tensors 
    [deform_tensors,deform_tensors_inv] = nonrigid.get_deformation_fields(shift_3D_total, par.regularize_deform_evol, size(rec_avg));

    %% SHOW SVD ANALYSIS OF VECTOR FIELD
    if debug()
        Nsvd = 3;
        cut_axis = 3; 
        upscale_arrows = 10;
        down_DVF = 4; % decimate it a bit to get easier image 
        nonrigid.show_deformation_field(rec_avg, deform_tensor_orig, ...
        Nps(1)/10 , Nsvd, binning,upscale_arrows, cut_axis,down_DVF)
    end
    
    smooth = max(par.min_smooth, smooth .* (par.min_smooth / par.max_smooth)^(1/par.Niter_registration));
    utils.verbose(2,'Current smoothing %g' , smooth)

    if par.rigid_correction
        %% estimate the 2D shift of the projection 
        %%%%%%%%%% TUNABLE PARAMETERS %%%%%%%%%%%%%%%%%%%%
        par.high_pass_filter = 0.05; % remove effect of residuums/bad ptycho connvergence , high value may get stack in local minima 
        %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
        par.showsorted = true; % selected angular order for alignment 
        par.valid_angles = 1:length(theta) > 0;   % use only this indices to do reconstruction (with respect to the time sorted theta)
        par.center_reconstruction = false; % keep the center of mass in center of rec. volume 
        par.align_horizontal = true; % horizontal alignment 
        par.align_vertical = true; % vertical alignment is worth usually only for laminography
        par.use_mask = false;    % apply support mask 
        par.mask_threshold = 0.001; % []; % empty == Otsu thresholding 
        par.use_localTV = false; % apply local TV 
        par.apply_positivity = true; % remove negative values 
        par.min_step_size  = 0.01;  %how much subpixel precision is needed
        par.max_iter = 100; % maximal number of iterations
        par.step_relaxation = 0.5; %% gradient decent step relaxation, (1 == full step), may be needed to avoid oscilations  
        % do not change 
        par.lamino_angle = 90;  % deg - laminography angle
        par.tilt_angle = 0; % tilt in the detector plane 
        par.vertical_scale = 1;
        par.horizontal_scale = 1;
        par.vert_range = 2:size(dphase,1)-1;
        par.binning = 1;
        par.unwrap_data_method = 'fft_1d';
        par.filter_type = 'ram-lak';
        par.refine_geometry = false;
        par.momentum_acceleration = false;  

        %%%%%%%%%%%%%%%%%%%%%%%%%%%
        %%%%%%%%%%%%%%%%%%%%%%%%%%%
        geom_shift = zeros(length(theta),2);
        
        % self consitency based alignment procedure based on the ASTRA toolbox 
        [geom_shift, par, rec, err] = tomo.align_tomo_consistency_linear(dphase,[], theta, Npix, ...
            geom_shift, par, 'deformation_fields', deform_tensors, 'inv_deformation_fields', deform_tensors_inv, ...
            'plot_results', false, 'verbose', min(0,utils.verbose()), 'CoR_offset', CoR_offset); 

        total_geom_shift(iter+1,:,:) = total_geom_shift(iter,:,:) + reshape(geom_shift,1,Ntheta,2);
        % imshift fft gives more precise shift than change of astra geometry
        dphase = gather(angle(utils.imshift_fft(exp(1i*gpuArray(dphase_0)), gather(squeeze(total_geom_shift(iter+1,:,:)))))); 
    end
    
    % unwrap data  after the shift is applied 
    sinogram = -tomo.block_fun(@math.unwrap2D_fft,dphase,2,par.air_gap); 
    

    %% deformation FBP 
    [rec_avg,rec_sub] = nonrigid.FBP_deform(sinogram, cfg, vectors,Bsize, deform_tensors_inv);
    %% object constraints 
    rec_avg = max(0, rec_avg .* rec_mask); % does not work well for our data 
    rec_sub = cellfun(@(x)(max(0,x .* rec_mask)), rec_sub, 'UniformOutput', false);

    
     sinogram_corr = zeros(Nlayers, Nw, Nangles, 'single');
     for ll = 1:Nblocks
        ids = 1+(ll-1)*Bsize:min(Ntheta, ll*Bsize); 
        sinogram_corr(:,:,ids) = gather(astra.Ax_partial(rec_avg,cfg, vectors(ids,:),1 , 'deformation_fields', deform_tensors{ll}));
     end

    if par.nonrigid_correction
        %% find 3D deformation 
        disp('Find 3D deformation ')
        x = gpuArray.ones(1,5);
        for ll = 1:Nblocks
            utils.progressbar(ll, Nblocks)
             % iterative solver 
            [shift_3D{ll},vol_err_tmp] = nonrigid.nonrigid_registration(rec_sub{ll},rec_avg, W_rec, par, smooth, 10);
            vol_err(ll, iter, :) = gather(vol_err_tmp([1,end]));
        end
        

        % update deformation field 
        for ll = 1:Nblocks
            for kk = 1:3
               shift_3D_total{ll}{kk}  = shift_3D_total{ll}{kk} + shift_3D{ll}{kk};
            end
        end

        %%  REGULARIZE RECONSTRUCTED FIELD -> correct boundary conditions 
        shift_3D_total=  nonrigid.regularize_3D_field(shift_3D_total);
    end
    if par.rigid_correction
        total_error(iter,:) = gather(err(end,:));
    end
    vert_range = reshape(ismember(1:Nlayers,ceil(Nlayers/10):ceil(Nlayers*9/10)),1,1,[]); 
    STD_score(iter) = std(reshape(rec_avg .* rec_mask .* vert_range,1,[])); 
    
    
    %% PLOT PROGRESS OF RECONSTRUCTION

    if toc(t_plot) > par.plot_every  % avoid plotting in every iteration, it is too slow ... 
        if par.rigid_correction
            nonrigid.show_rigid_corrections(rec,  sinogram, err,total_geom_shift, theta, iter, par); 
        end
        
        if par.nonrigid_correction
            figure(5465)
            subplot(2,2,1)
            loglog(vol_err(:,:,1)')
            title('Volume error (<V> - V_i)')
            axis tight 
            subplot(2,2,2)
            semilogy(vol_err(:,:,1), '.')
            title('Volume error (<V> - V_i)''')
            axis tight 
            subplot(2,2,3)
            hold on
            plot(total_error)
            plot(mean(total_error,2), 'k', 'LineWidth', 3);
            hold off 
            grid on 
            axis tight
            xlim([1,iter+1])
            set(gca, 'xscale', 'log')
            set(gca, 'yscale', 'log')
            title('Projection-domain error')
            xlabel('Iteration')
            ylabel('Mean abs error')
            subplot(2,2,4)
            plot((STD_score-min(STD_score)) ./ (max(STD_score) - min(STD_score)))
            legend({'STD'})
            title('STD of reconstruction (higher is better)')
            set(gca, 'xscale', 'log')
            xlim([1,iter+1])
            plotting.suptitle('Evolution of nonrigid tomo reconstruction')

            figure(1)
            b_id = Nblocks;
            [deform_tensors] = nonrigid.get_deformation_fields(shift_3D_total, par.regularize_deform_evol, size(rec_avg));
            for k = 1:3
                subplot(use_artif_data+2,3,k)
                imagesc(shift_3D{b_id}{k}(:,:,ceil(Nps(3)/2)))
                colorbar
                axis image 
                colormap bone 
                set(gca, 'xtick', []);set(gca, 'ytick', []);
                if k == 1; ylabel('Deformation update'); end
                ax = {'x', 'y', 'y'}; 
                title(sprintf('DVF along axis %s', ax{k}))
                subplot(use_artif_data+2,3,3+k)
                imagesc(deform_tensors{b_id}{1,k}(:,:,ceil(Nps(3)/2)))
                colorbar
                set(gca, 'xtick', []);set(gca, 'ytick', []);
                axis image 
                if k == 1; ylabel('Total deformation'); end

                if use_artif_data
                    subplot(use_artif_data+2,3,6+k)
                    imagesc(deform_tensor_orig{b_id}{1,k}(:,:,ceil(Nps(3)/2)) / par.binning )
                    colorbar
                    set(gca, 'xtick', []);set(gca, 'ytick', []);
                    axis image 
                    if k == 1; ylabel('Ideal deformation'); end
                end
            end
            suptitle('Reconstructed DVF')
        
        end

        drawnow 
        t_plot = tic; 

    end
    
end

%% gather results from GPU 
for kk = 1:3
    for ll = 1:Nblocks
        shift_3D_total{ll}{kk} = gather(shift_3D_total{ll}{kk});
    end
end
vol_err = gather(vol_err); 


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%   reconstruction of deformation done %%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%% ANALYSIS OF LOW RESOLUTION PREVIEW %%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

%% SHOW ANIMATION OF PROJECTIONS 
show_derivative = true; 

nonrigid.show_sinograms(rec_avg,dphase, vectors,cfg, theta,  shift_3D_total, par, show_derivative );
plotting.suptitle('Low resolution sinograms')



%% SHOW SVD ANALYSIS OF VECTOR FIELD 
Nsvd = 2;
cut_axis = 3; 
upscale_arrows = 5;
down_DVF = 2; % decimate it a bit to get easier image 

[deform_tensors,deform_tensors_inv] = nonrigid.get_deformation_fields(shift_3D_total, par.regularize_deform_evol, size(rec_avg));

nonrigid.show_deformation_field(rec_avg, deform_tensors, ...
   Nps(1)/8, Nsvd, binning,upscale_arrows, cut_axis, down_DVF)



%%  COMPARE RECONSTRUCTION QUALITY OF THE SMALL TOMOGRAMS

sinogram = -tomo.block_fun(@math.unwrap2D_fft,dphase,2,par.air_gap); 

Niter_SART = 50;
nonrigid.show_reconstruction_quality(sinogram, cfg, vectors, shift_3D_total, par.regularize_deform_evol);
plotting.suptitle('Low resolution reconstructions')

%% SAVE RECONSTRUCTED DEFORMATION FIELDS %%%%


if ~debug() && ~isempty(par.output_folder)
    save(fullfile(par.output_folder, ['shift_3D_total_', sample_name, '_nblocks_', num2str(Nblocks)]), 'shift_3D_total')
end


if use_artif_data; return; end


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%% RECONSTRUCT ORIGINAL DATA USING ESTIMATED DVF AND PROJECTIONS SHIFTS  %%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%



import plotting.*
import math.*
import utils.*

%  initialize 

Nblocks = length(shift_3D_total);
[Nlayers_full, Nw_full,~] = size(dphase_0_full); 
Npix = ceil(Nw_full/32)*32; 
Npix = [Npix, Npix, Nlayers_full];
par.air_gap = par.air_gap*binning; 


[cfg_full, vectors_full] = ...
    astra.ASTRA_initialize(Npix,[Npix(3),Nw_full],theta);



for ll = 1:Nblocks 
    for jj=1:3
        shift_3D_total_full{ll}{jj} =  gather(shift_3D_total{ll}{jj} * binning);
    end
end

Ntheta=length(theta);
[deform_tensors_full, inv_deform_tensors_full]= nonrigid.get_deformation_fields(shift_3D_total_full, par.regularize_deform_evol, Npix);

par.deformation_vector_field = shift_3D_total_full; 


if par.rigid_correction
    last_iteration = find(any(any(total_geom_shift,2),3),1, 'last'); 
    sino_shift = squeeze(total_geom_shift(last_iteration,:,:) * binning);
else
    sino_shift = zeros(Ntheta,2);
end

disp('Apply shift on data')
dphase_0_full = tomo.block_fun(@(x,shift)(angle(utils.imshift_fft(exp(1i*x), shift))) , dphase_0_full, sino_shift);
% unwrap data 
% sinogram_full = -tomo.block_fun(@math.unwrap2D_fft,dphase_0_full,2,par.air_gap); 

object_0_full = tomo.block_fun(@(x,shift)(utils.imshift_fft(x, shift)), object_0_full, sino_shift);
ROI{1}=[1:size(object_0_full,1)];
ROI{2}=[1:size(object_0_full,2)];
sinogram_full = tomo.unwrap_2D_bootstrap(object_0_full, theta ,par, 5, ROI); 

%%  COMPARE RECONSTRUCTION QUALITY OF THE FULL TOMOGRAMS


[rec_0_full, rec_FBP_full, rec_SART_full] = nonrigid.show_reconstruction_quality(sinogram_full, cfg_full, vectors_full, shift_3D_total_full, par.regularize_deform_evol);

[~, W_rec] = nonrigid.get_mask(rec_0_full, par.mask_threshold, par.mask_dilate); 

mask = sum(rec_0_full,3) > max(max(sum(rec_0_full,3))/10); 


%% show sinograms 

rec_FBP_full = nonrigid.FBP_deform(sinogram_full, cfg_full, vectors_full, Bsize, inv_deform_tensors_full );

nonrigid.show_sinograms(rec_FBP_full,dphase_0_full, vectors_full,cfg_full, theta,  shift_3D_total_full, par , true)
plotting.suptitle('Full resolution reconstructions')


%% save results 

if isfield(par, 'factor')
    % calculate complex refractive index 
    tomogram_delta = rec_SART_full*par.factor;
end

%% Save Tomogram (delta)
% Note saving tomogram in delta (index of refraction)
% To get phase tomogram   = tomogram_delta/par.factor
% To get electron density tomogram in [e/A^3]   = tomogram_delta*par.factor_edensity
% If you saved before you may clear 
par.scans_string = '';
par.freq_scale = 1; 
par.force_overwrite = false;
    
if ~debug()
    tomo.save_tomogram(tomogram_delta, par, 'delta', [], theta, '_nonrigid_SART')
end





%% Save Tiff files for 3D visualization with external program
%%%%%%%%%%%%%%%%%%%%%%%%%
%%% Edit this section %%%
%%%%%%%%%%%%%%%%%%%%%%%%%
par.save_as_stack = false; 
par.tiff_compression = 'none';
par.tiff_subfolder_name = ['TIFF_delta_' par.filter_type '_freqscl_' sprintf('%0.2f',par.freq_scale), '_nonrigid_SART'];
par.name_prefix = 'tomo_delta';
%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%
if  ~debug()
	  tomo.save_as_tiff(tomogram_delta, par,'nonrigid')
end

disp('======== FINISHED =========')



%*-----------------------------------------------------------------------*
%|                                                                       |
%|  Except where otherwise noted, this work is licensed under a          |
%|  Creative Commons Attribution-NonCommercial-ShareAlike 4.0            |
%|  International (CC BY-NC-SA 4.0) license.                             |
%|                                                                       |
%|  Copyright (c) 2017 by Paul Scherrer Institute (http://www.psi.ch)    |
%|                                                                       |
%|       Author: CXS group, PSI                                          |
%*-----------------------------------------------------------------------*
% You may use this code with the following provisions:
%
% If the code is fully or partially redistributed, or rewritten in another
%   computing language this notice should be included in the redistribution.
%
% If this code, or subfunctions or parts of it, is used for research in a 
%   publication or if it is fully or partially rewritten for another 
%   computing language the authors and institution should be acknowledged 
%   in written form in the publication: “Data processing was carried out 
%   using the “cSAXS matlab package” developed by the CXS group,
%   Paul Scherrer Institut, Switzerland.” 
%   Variations on the latter text can be incorporated upon discussion with 
%   the CXS group if needed to more specifically reflect the use of the package 
%   for the published work.
%
% A publication that focuses on describing features, or parameters, that
%    are already existing in the code should be first discussed with the
%    authors.
%   
% This code and subroutines are part of a continuous development, they 
%    are provided “as they are” without guarantees or liability on part
%    of PSI or the authors. It is the user responsibility to ensure its 
%    proper use and the correctness of the results.
