%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%% EDIT HERE:
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% where the data is saved (Data10, Data20, p-account)
BasePath = '~/Data10/';
e_account = beamline.identify_eaccount;
file_flag='';              % suffix used in radial integration folder
                           % Example: '_Pilatus2M_at_2m'; 
                           % Leave empty '' for default folder names                   
mica_thickness_xbox=2;     % [micron]
mica_thickness_FT=5;       % [micron]
mylar_thickness=300;       % [micron] exit FT window is assumed to be mylar!
%_____________________________________________________________
% needed for flux calculation using bim2
use_bim2 = 1;
t_d1=5.04;              % thickness of Si sensor [microns]
% Remember to activate Elletra
scan_dark=173;          % scan with closed shutter (set offset in Elettra device such that counts are almost zero)
scan=174;               % scan with open shutter and BIM2 diode in the beam
air_path = 7;           % air path distance between the exposure box window and the sample in mm
Elettra_bitmode = 24;
Elettra_range = 2.5e-3; % [A]
%_____________________________________________________________
% needed for flux calculation using the glassy carbon L14
use_GCL14 = 1;
Glassy_carbon=175;      % the scan number of glassy carbon
Air = 174;              %scan number of air scattering
DB = Air;               %direct beam, use air scattering in case there is no direct beam measurement
%parameters for calculating the flux
DetDist_mm = 2.148e3;   % [mm]
used_bs = 'diode';      % Spec counter used for transmission measurement
                        % Typically 'diode' or 'cyb' at the beamstop 
%_____________________________________________________________
% also scale WAXS data?
use_WAXS = 0; % 1 to scale the WAXS data, 0 not used
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%% calculations
if use_bim2
    % calculation
    %read the spec files
    S=io.spec_read(BasePath,'ScanNr',scan);
    S_dark=io.spec_read(BasePath,'ScanNr',scan_dark);
    
    %read the energy in keV
    energy = S_dark.mokev;
    
    %constants
    epsilon_Si=3.66;  % energy required to generate an electron-hole pair in Si [eV]
    e_charge=1.6e-19; %electron change [C]
    
    % from http://henke.lbl.gov/optical_constants/filter2.html, considering
    % mica KAl3Si3O11.8H1.8F0.2 Density=2.76 Thickness= mica_thickness_xbox
    % For mica window at the exposure box exit
    [val, ~ ] = utils.get_fil_trans( 'mica', [energy, energy+0.01], mica_thickness_xbox);
    trans_factor_d1 = val(1,2);
    
    %calculate the transmission of the air path
    % between the exposure box exit and the sample
    %from http://henke.lbl.gov/optical_constants/gastrn2.html considering
    %air N1.562O.42C.0003Ar.0094	Pressure=760 torr.	T = 297 K, Path =
    %air_path/10 [cm]
    [ val, ~ ] = utils.get_gas_trans('air', [energy, energy+0.01], air_path/10,760, 297 );
    ap_trans = val(1,2);
    
    
    % from http://henke.lbl.gov/optical_constants/filter2.html, considering
    % Silicon Density=2.33 Thickness= 10. microns
    % contributes positively to the calculated flux
    %t_d1=10;     % thickness Si sensor in microns
    [ val, ~ ] = utils.get_fil_trans( 'Si', [energy,energy+0.01],t_d1);
    trans_si = val(1,2);
    
    %calculate the counts difference between open and closed shutter
    counts_d1 = mean(S.transd) - mean(S_dark.transd); %photons
    fprintf('The average counts at the  transmission diode is %7.2f photons \n',counts_d1)
    
    %calculate the current
    curr_d1 = counts_d1 * Elettra_range/(2^(Elettra_bitmode-1));  % [A]
    fprintf('The current at the transmission diode is %7.2f microA \n',curr_d1*1e6)
    
    %calculate the flux
      flux_BIM2= ((curr_d1 * epsilon_Si)/(e_charge* (energy*1e3) *(1-trans_si)))*(trans_factor_d1 * ap_trans);
    fprintf('The flux based on bim2 is %7.2e photons/s \n',flux_BIM2)
end

%%
if use_GCL14
    
    thickness_L14 = 0.1; % Thickness of glassy carbon L14 standard [cm]
    
    %calculate the relative transmittance
    %load the diode value for the air
    S_air = io.spec_read(BasePath,'ScanNr',Air);
    %scale in case the exposure times are different
    exp_time = S_air.sec(1,1);
    scale_air = 1/exp_time;
    
    %read the spec data for the glassy carbon sample
    S = io.spec_read(BasePath,'ScanNr',Glassy_carbon);
    energy = S.mokev; % in kev
    pixel_size = 0.172; % in mm
    %normalize the measurement to the correct
    exp_time = S.sec(1,1);
    scale_GC = 1/exp_time;
    
    %calculation of transmittance
  
    transmittance = (mean(S.(used_bs))/mean(S.bpm4i))/(mean(S_air.(used_bs))/mean(S_air.bpm4i));
    
    %radial integration file for GC
    filename_GC = fullfile(BasePath,'analysis',sprintf('radial_integration%s',file_flag),sprintf('%s_1_%05d_00000_00000_integ.h5',e_account,Glassy_carbon));
    Data_GC = io.HDF.hdf5_load(filename_GC);
    I_GC_m = mean(Data_GC.I_all,3);
    I_GC_m = sum(I_GC_m.*Data_GC.norm_sum,2)./sum(Data_GC.norm_sum, 2);
    %q_GC_m = utils.pixel_to_q(Data_GC.radius, pixel_size, DetDist_mm, energy);
    q_GC_m = Data_GC.q;
    
    %estimate the direct beam measurement
    S_DB = io.spec_read(BasePath,'ScanNr',DB);
    %scale in case the exposure times are different
    exp_time = S_DB.sec(1,1);
    scale_DB = 1/exp_time;
    
    
    %radial integration file for direct beam
    filename_DB = fullfile(BasePath,'analysis',sprintf('radial_integration%s',file_flag),sprintf('%s_1_%05d_00000_00000_integ.h5',e_account,DB));
    Data_DB = io.HDF.hdf5_load(filename_DB);
    I_DB= mean(Data_DB.I_all,3);
    I_DB = sum(I_DB.*Data_DB.norm_sum,2)./sum(Data_DB.norm_sum, 2);
    
    %normalize by the exposure time
    I_GC_m = ((((I_GC_m*scale_GC)*1/transmittance )-(I_DB*scale_DB))*1/thickness_L14);
    
    %compare to calibrated standard
    %values provided for the calibrated Glassy carbon sample
    %where is this coming from? The dat file in +beamline in master has
    %data only up to 1.172 A-1
%     q_GC_c = [0.0120496340976507;0.0125044551874072;0.0129732051637538;0.0134583002623949;0.0139593140871214;0.0144769572942580;0.0150120826715035;0.0155646902169131;0.0161354905862074;0.0167244837770547;0.0173329489709293;0.0179618810858233;0.0186112801185458;0.0192811460656564;0.0199724738441620;0.0206865426336400;0.0214223575088364;0.0221826189635588;0.0229661899401404;0.0237744917475841;0.0246092299570281;0.0254699781665768;0.0263573048941064;0.0272720629196088;0.0282159578107348;0.0291889895567488;0.0301914424089283;0.0312250219314847;0.0322902966359584;0.0333882614266597;0.0345199112062792;0.0356860987444304;0.0368869661524214;0.0381246453778608;0.0393994206591611;0.0407117183640269;0.0420638125635520;0.0434554189640091;0.0448893801546903;0.0463648433112569;0.0478845088842486;0.0494488032221519;0.0510572998839037;0.0527151155326772;0.0544201181456095;0.0561750081476676;0.0579803539937366;0.0598374347869336;0.0617482402768343;0.0637136231591258;0.0657351467748547;0.0678138059336549;0.0699520167405197;0.0721506318538254;0.0744114988336479;0.0767360388361561;0.0791253887439922;0.0815812539462713;0.0841040606468480;0.0867000623577208;0.0893671269066933;0.0921071017413062;0.0949225449342727;0.0978154460147494;0.100787225969034;0.103839874273604;0.106976091018953;0.110196728587036;0.113505339771704;0.116902066242041;0.120391029223623;0.123972796689442;0.127650494872454;0.131426112929190;0.135299792309430;0.139280344160401;0.143364214499320;0.147555239889708;0.151856261927069;0.156270548500290;0.160799093399002;0.165445448584523;0.170213165908177;0.175104375861299;0.180120640330274;0.185266789926598;0.190545381136233;0.195958259692354;0.201509545135676;0.207201793489021;0.213037134240793;0.219023239380533;0.225158684973226;0.231448584401029;0.237896203237904;0.244506227971290;0.251280076107924;0.258222433557780;0.265337275310852;0.272627723322335;0.280098746656204;0.287753608607507;0.290087652062856;0.293304736269482;0.296521618321574;0.299738296022361;0.302954767175882;0.306171029586988;0.309387081061351;0.312602919405477;0.315818542426710;0.319033947933244;0.322249133734126;0.325464097639273;0.328678837459470;0.331893351006389;0.335107636092589;0.338321690531530;0.341535512137577;0.344749098726013;0.347962448113043;0.351175558115807;0.354388426552383;0.357601051241800;0.360813430004044;0.364025560660067;0.367237441031796;0.370449068942139;0.373660442214998;0.376871558675271;0.380082416148865;0.383293012462703;0.386503345444732;0.389713412923933;0.392923212730325;0.396132742694977;0.399342000650017;0.402550984428637;0.405759691865102;0.408968120794761;0.412176269054052;0.415384134480512;0.418591714912783;0.421799008190624;0.425006012154916;0.428212724647670;0.431419143512038;0.434625266592318;0.437831091733966;0.441036616783597;0.444241839589003;0.447446757999153;0.450651369864203;0.453855673035507;0.457059665365622;0.460263344708318;0.463466708918585;0.466669755852640;0.469872483367936;0.473074889323173;0.476276971578300;0.479478727994529;0.482680156434336;0.485881254761478;0.489082020840992;0.492282452539209;0.495482547723759;0.498682304263582;0.501881720028930;0.505080792891380;0.508279520723843;0.511477901400565;0.514675932797143;0.517873612790525;0.521070939259026;0.524267910082328;0.527464523141494;0.530660776318971;0.533856667498600;0.537052194565627;0.540247355406704;0.543442147909900;0.546636569964711;0.549830619462064;0.553024294294327;0.556217592355316;0.559410511540301;0.562603049746017;0.565795204870670;0.568986974813943;0.572178357477005;0.575369350762521;0.578559952574654;0.581750160819079;0.584939973402985;0.588129388235086;0.591318403225627;0.594507016286392;0.597695225330712;0.600883028273470;0.604070423031114;0.607257407521657;0.610443979664689;0.613630137381386;0.616815878594512;0.620001201228432;0.623186103209116;0.626370582464145;0.629554636922725;0.632738264515686;0.635921463175494;0.639104230836259;0.642286565433740;0.645468464905351;0.648649927190174;0.651830950228960;0.655011531964139;0.658191670339828;0.661371363301837;0.664550608797674;0.667729404776558;0.670907749189420;0.674085639988915;0.677263075129425;0.680440052567068;0.683616570259708;0.686792626166954;0.689968218250177;0.693143344472510;0.696318002798857;0.699492191195902;0.702665907632112;0.705839150077748;0.709011916504869;0.712184204887341;0.715356013200841;0.718527339422871;0.721698181532753;0.724868537511648;0.728038405342554;0.731207783010319;0.734376668501645;0.737545059805092;0.740712954911092;0.743880351811948;0.747047248501847;0.750213642976862;0.753379533234962;0.756544917276018;0.759709793101807;0.762874158716023;0.766038012124281;0.769201351334123;0.772364174355027;0.775526479198411;0.778688263877642;0.781849526408042;0.785010264806893;0.788170477093445;0.791330161288921;0.794489315416526;0.797647937501451;0.800806025570882;0.803963577654003;0.807120591782006;0.810277065988094;0.813432998307490;0.816588386777444;0.819743229437236;0.822897524328184;0.826051269493653;0.829204462979055;0.832357102831862;0.835509187101608;0.838660713839898;0.841811681100411;0.844962086938909;0.848111929413243;0.851261206583356;0.854409916511293;0.857558057261207;0.860705626899361;0.863852623494139;0.866999045116048;0.870144889837729;0.873290155733956;0.876434840881649;0.879578943359877;0.882722461249862;0.885865392634988;0.889007735600807;0.892149488235043;0.895290648627597;0.898431214870558;0.901571185058203;0.904710557287006;0.907849329655645;0.910987500265001;0.914125067218175;0.917262028620484;0.920398382579469;0.923534127204906;0.926669260608805;0.929803780905419;0.932937686211250;0.936070974645051;0.939203644327838;0.942335693382891;0.945467119935758;0.948597922114265;0.951728098048522;0.954857645870923;0.957986563716156;0.961114849721207;0.964242502025366;0.967369518770233;0.970495898099721;0.973621638160064;0.976746737099822;0.979871193069885;0.982995004223480;0.986118168716174;0.989240684705883;0.992362550352873;0.995483763819770;0.998604323271560;1.00172422687560;1.00484347280162;1.00796205922172;1.01107998431039;1.01419724624452;1.01731384320337;1.02042977336862;1.02354503492435;1.02665962605703;1.02977354495557;1.03288678981129;1.03599935881792;1.03911125017165;1.04222246207107;1.04533299271724;1.04844284031363;1.05155200306620;1.05466047918332;1.05776826687586;1.06087536435712;1.06398176984290;1.06708748155144;1.07019249770348;1.07329681652225;1.07640043623343;1.07950335506523;1.08260557124835;1.08570708301598;1.08880788860381;1.09190798625008;1.09500737419550;1.09810605068332;1.10120401395931;1.10430126227179;1.10739779387158;1.11049360701207;1.11358869994917;1.11668307094135;1.11977671824962;1.12286964013756;1.12596183487130;1.12905330071955;1.13214403595357;1.13523403884722;1.13832330767690;1.14141184072162;1.14449963626298;1.14758669258515;1.15067300797492;1.15375858072165;1.15684340911734;1.15992749145657;1.16301082603654;1.16609341115706;1.16917524512058;1.17225632623216;1.17533665279948;1.17841622313287;1.18149503554529;1.18457308835233;1.18765037987225;1.19072690842594;1.19380267233694;1.19687766993146;1.19995189953835;1.20302535948915;1.20609804811805;1.20916996376192;1.21224110476031;1.21531146945542;1.21838105619218;1.22144986331817;1.22451788918368;1.22758513214169;1.23065159054788;1.23371726276062;1.23678214714101;1.23984624205283;1.24290954586259;1.24597205693952;1.24903377365556;1.25209469438539;1.25515481750638;1.25821414139868;1.26127266444514;1.26433038503136;1.26738730154567;1.27044341237917;1.27349871592567;1.27655321058177;1.27960689474680;1.28265976682286;1.28571182521480;1.28876306833024;1.29181349457958;1.29486310237597;1.29791189013534;1.30095985627641;1.30400699922068;1.30705331739241;1.31009880921867;1.31314347312932;1.31618730755700;1.31923031093715;1.32227248170803;1.32531381831067;1.32835431918892;1.33139398278945;1.33443280756172;1.33747079195802;1.34050793443346;1.34354423344595;1.34657968745625;1.34961429492792;1.35264805432736;1.35568096412381;1.35871302278933;1.36174422879884;1.36477458063007;1.36780407676360;1.37083271568289;1.37386049587419;1.37688741582665;1.37991347403226;1.38293866898585;1.38596299918512;1.38898646313064;1.39200905932582;1.39503078627698;1.39805164249326;1.40107162648669;1.40409073677219;1.40710897186754;1.41012633029340;1.41314281057332;1.41615841123372;1.41917313080391;1.42218696781611;1.42519992080540;1.42821198830976;1.43122316887009;1.43423346103016;1.43724286333665;1.44025137433914;1.44325899259012;1.44626571664498;1.44927154506203;1.45227647640248;1.45528050923045;1.45828364211298;1.46128587362004;1.46428720232451;1.46728762680218;1.47028714563178;1.47328575739495;1.47628346067627;1.47928025406325;1.48227613614633;1.48527110551887;1.48826516077719;1.49125830052052;1.49425052335104;1.49724182787389;1.50023221269712;1.50322167643175;1.50621021769172;1.50919783509396;1.51218452725830;1.51517029280755;1.51815513036748;1.52113903856680;1.52412201603717;1.52710406141322;1.53008517333255;1.53306535043570;1.53604459136618;1.53902289477047;1.54200025929801;1.54497668360121;1.54795216633545;1.55092670615908;1.55390030173342;1.55687295172277;1.55984465479439;1.56281540961853;1.56578521486842;1.56875406922025;1.57172197135321;1.57468891994946;1.57765491369415;1.58061995127541;1.58358403138436;1.58654715271510;1.58950931396471;1.59247051383329;1.59543075102390;1.59839002424259;1.60134833219843;1.60430567360347;1.60726204717273;1.61021745162427;1.61317188567911;1.61612534806130;1.61907783749785;1.62202935271880;1.62497989245718;1.62792945544904;1.63087804043339;1.63382564615229;1.63677227135077;1.63971791477690;1.64266257518171;1.64560625131929;1.64854894194669;1.65149064582400;1.65443136171431;1.65737108838372;1.66030982460134;1.66324756913928;1.66618432077270;1.66912007827973;1.67205484044154;1.67498860604230;1.67792137386920;1.68085314271246;1.68378391136529;1.68671367862394;1.68964244328766;1.69257020415873;1.69549696004244;1.69842270974710;1.70134745208405;1.70427118586765;1.70719390991525;1.71011562304725;1.71303632408708;1.71595601186116;1.71887468519895;1.72179234293293;1.72470898389861;1.72762460693450;1.73053921088216;1.73345279458615;1.73636535689408;1.73927689665656;1.74218741272723;1.74509690396276;1.74800536922285;1.75091280737020;1.75381921727057;1.75672459779271;1.75962894780841;1.76253226619250;1.76543455182281;1.76833580358020;1.77123602034858;1.77413520101486;1.77703334446897;1.77993044960388;1.78282651531559;1.78572154050312;1.78861552406850;1.79150846491680;1.79440036195611;1.79729121409756;1.80018102025528;1.80306977934643;1.80595749029120;1.80884415201282;1.81172976343751;1.81461432349454;1.81749783111618;1.82038028523776;1.82326168479758;1.82614202873702;1.82902131600044;1.83189954553523;1.83477671629182;1.83765282722364;1.84052787728715;1.84340186544183;1.84627479065018;1.84914665187771;1.85201744809296;1.85488717826749;1.85775584137587;1.86062343639569;1.86348996230756;1.86635541809510;1.86921980274495;1.87208311524676;1.87494535459320;1.87780651977996;1.88066660980573;1.88352562367222;1.88638356038415;1.88924041894925;1.89209619837827;1.89495089768496;1.89780451588608;1.90065705200139;1.90350850505369;1.90635887406874;1.90920815807536;1.91205635610532;1.91490346719343;1.91774949037749;1.92059442469832;1.92343826919972;1.92628102292850;1.92912268493447;1.93196325427044;1.93480272999222;1.93764111115861;1.94047839683142;1.94331458607543;1.94614967795845;1.94898367155125;1.95181656592761;1.95464836016430;1.95747905334108;1.96030864454069;1.96313713284887;1.96596451735434;1.96879079714881;1.97161597132697;1.97444003898649;1.97726299922804;1.98008485115525;1.98290559387474;1.98572522649611;1.98854374813193;1.99136115789776;1.99417745491212;1.99699263829651;1.99980670717540;2.00261966067624;2.00543149792944;2.00824221806838;2.01105182022941;2.01386030355184;2.01666766717795;2.01947391025299;2.02227903192515;2.02508303134560;2.02788590766846;2.03068766005082;2.03348828765269;2.03628778963708;2.03908616516993;2.04188341342012;2.04467953355950;2.04747452476286;2.05026838620794;2.05306111707542;2.05585271654892;2.05864318381502;2.06143251806322;2.06422071848596;2.06700778427864;2.06979371463957;2.07257850877001;2.07536216587413;2.07814468515905;2.08092606583482;2.08370630711441;2.08648540821371;2.08926336835154;2.09204018674965;2.09481586263269;2.09759039522824;2.10036378376680;2.10313602748177;2.10590712560949;2.10867707738918;2.11144588206298;2.11421353887595;2.11698004707603;2.11974540591410;2.12250961464390;2.12527267252209;2.12803457880823;2.13079533276478;2.13355493365708;2.13631338075336;2.13907067332476;2.14182681064528;2.14458179199184;2.14733561664421;2.15008828388506;2.15283979299994;2.15559014327727;2.15833933400834;2.16108736448734;2.16383423401131;2.16657994188015;2.16932448739666;2.17206786986647;2.17481008859809;2.17755114290289;2.18029103209511;2.18302975549182;2.18576731241296;2.18850370218132;2.19123892412255;2.19397297756512;2.19670586184038;2.19943757628250;2.20216812022849;2.20489749301821;2.20762569399436;2.21035272250246;2.21307857789086;2.21580325951075;2.21852676671615;2.22124909886389;2.22397025531364;2.22669023542787;2.22940903857189;2.23212666411381;2.23484311142455;2.23755837987785;2.24027246885026;2.24298537772112;2.24569710587260;2.24840765268964;2.25111701755999;2.25382519987421;2.25653219902564;2.25923801441041;2.26194264542744;2.26464609147844;2.26734835196789;2.27004942630307;2.27274931389402;2.27544801415358;2.27814552649733;2.28084185034365;2.28353698511367;2.28623093023130;2.28892368512319;2.29161524921877;2.29430562195022;2.29699480275248;2.29968279106324;2.30236958632294;2.30505518797475;2.30773959546461;2.31042280824119;2.31310482575591;2.31578564746289;2.31846527281903;2.32114370128393;2.32382093231993;2.32649696539210;2.32917179996822;2.33184543551879;2.33451787151703;2.33718910743889;2.33985914276300;2.34252797697073;2.34519560954614;2.34786203997599;2.35052726774974;2.35319129235956;2.35585411330031;2.35851573006953;2.36117614216746;2.36383534909702;2.36649335036382;2.36915014547614;2.37180573394495;2.37446011528388;2.37711328900924;2.37976525464002;2.38241601169784;2.38506555970702;2.38771389819453;2.39036102668997;2.39300694472564;2.39565165183645;2.39829514755998;2.40093743143644;2.40357850300870;2.40621836182226;2.40885700742525;2.41149443936843;2.41413065720522;2.41676566049162;2.41939944878629;2.42203202165049;2.42466337864812;2.42729351934567;2.42992244331226;2.43255015011959;2.43517663934201;2.43780191055643;2.44042596334239;2.44304879728200;2.44567041195998;2.44829080696363;2.45090998188285;2.45352793631011;2.45614466984046;2.45876018207154;2.46137447260355;2.46398754103927;2.46659938698405;2.46921001004578;2.47181940983495;2.47442758596457;2.47703453805024;2.47964026571008;2.48224476856477;2.48484804623755;2.48745009835419;2.49005092454298;2.49265052443478;2.49524889766296;2.49784604386342;2.50044196267460;2.50303665373744;2.50563011669542;2.50822235119452;2.51081335688325;2.51340313341261;2.51599168043611;2.51857899760978;2.52116508459213;2.52374994104418;2.52633356662942;2.52891596101385;2.53149712386596;2.53407705485670;2.53665575365952;2.53923321995034;2.54180945340753;2.54438445371198;2.54695822054699;2.54953075359836;2.55210205255433;2.55467211710562;2.55724094694536;2.55980854176916;2.56237490127508;2.56494002516360;2.56750391313765;2.57006656490260;2.57262798016624;2.57518815863879;2.57774710003291;2.58030480406367;2.58286127044855;2.58541649890745;2.58797048916270;2.59052324093900;2.59307475396350;2.59562502796570;2.59817406267754;2.60072185783333;2.60326841316978;2.60581372842598;2.60835780334341;2.61090063766592;2.61344223113975;2.61598258351350;2.61852169453814;2.62105956396702;2.62359619155583;2.62613157706264;2.62866572024786;2.63119862087425;2.63373027870693;2.63626069351336;2.63878986506334;2.64131779312900;2.64384447748481;2.64636991790758;2.64889411417643;2.65141706607280;2.65393877338047;2.65645923588553;2.65897845337636;2.66149642564367;2.66401315248048;2.66652863368210;2.66904286904613;2.67155585837247;2.67406760146333;2.67657809812318;2.67908734815878;2.68159535137919;2.68410210759571;2.68660761662195;2.68911187827376;2.69161489236927;2.69411665872887;2.69661717717520;2.69911644753317;2.70161446962991;2.70411124329485;2.70660676835960;2.70910104465805;2.71159407202633;2.71408585030277;2.71657637932796;2.71906565894470;2.72155368899800;2.72404046933512;2.72652599980551;2.72901028026082;2.73149331055495;2.73397509054395;2.73645562008610;2.73893489904188;2.74141292727394;2.74388970464714;2.74636523102850;2.74883950628724;2.75131253029475;2.75378430292460;2.75625482405253;2.75872409355642;2.76119211131634;2.76365887721452;2.76612439113533;2.76858865296529;2.77105166259308;2.77351341990951;2.77597392480755;2.77843317718228;2.78089117693092;2.78334792395284;2.78580341814951;2.78825765942452;2.79071064768361;2.79316238283458;2.79561286478740;2.79806209345409;2.80051006874882;2.80295679058782;2.80540225888944;2.80784647357411;2.81028943456436;2.81273114178478;2.81517159516205;2.81761079462495;2.82004874010429;2.82248543153299;2.82492086884600;2.82735505198035;2.82978798087512;2.83221965547146;2.83465007571253;2.83707924154359;2.83950715291190;2.84193380976677;2.84435921205955;2.84678335974362;2.84920625277438;2.85162789110926;2.85404827470771;2.85646740353118;2.85888527754316;2.86130189670913;2.86371726099656;2.86613137037495;2.86854422481579;2.87095582429254;2.87336616878068;2.87577525825766;2.87818309270290;2.88058967209781;2.88299499642579;2.88539906567218;2.88780187982431;2.89020343887145;2.89260374280485;2.89500279161771;2.89740058530517;2.89979712386432;2.90219240729421;2.90458643559581;2.90697920877203;2.90937072682773;2.91176098976967;2.91414999760655;2.91653775034900;2.91892424800954;2.92130949060263;2.92369347814463;2.92607621065379;2.92845768815029;2.93083791065618;2.93321687819543;2.93559459079388;2.93797104847927;2.94034625128120;2.94272019923118;2.94509289236258;2.94746433071064;2.94983451431246;2.95220344320703;2.95457111743516;2.95693753703955;2.95930270206474;2.96166661255712;2.96402926856492;2.96639067013821;2.96875081732891;2.97110971019076;2.97346734877933;2.97582373315202;2.97817886336805;2.98053273948846;2.98288536157610;2.98523672969564;2.98758684391354;2.98993570429808;2.99228331091932;2.99462966384914;2.99697476316119;2.99931860893091;3.00166120123554;3.00400254015408;3.00634262576731;3.00868145815780;3.01101903740987;3.01335536360961;3.01569043684487;3.01802425720525;3.02035682478213;3.02268813966861;3.02501820195955;3.02734701175154;3.02967456914293;3.03200087423378;3.03432592712591;3.03664972792282;3.03897227672979;3.04129357365378;3.04361361880348;3.04593241228928;3.04824995422329;3.05056624471932;3.05288128389289;3.05519507186120;3.05750760874314;3.05981889465932;3.06212892973201;3.06443771408515;3.06674524784439;3.06905153113703;3.07135656409205;3.07366034684009;3.07596287951346;3.07826416224614;3.08056419517373;3.08286297843351;3.08516051216440;3.08745679650696;3.08975183160340;3.09204561759757;3.09433815463492;3.09662944286256;3.09891948242923;3.10120827348526;3.10349581618262;3.10578211067490;3.10806715711728;3.11035095566656;3.11263350648114;3.11491480972101;3.11719486554776;3.11947367412459;3.12175123561626;3.12402755018913;3.12630261801114;3.12857643925179;3.13084901408218;3.13312034267495;3.13539042520432;3.13765926184609;3.13992685277758;3.14219319817769;3.14445829822687;3.14672215310710;3.14898476300192;3.15124612809639;3.15350624857714;3.15576512463230;3.15802275645154;3.16027914422605;3.16253428814855;3.16478818841326;3.16704084521594;3.16929225875384;3.17154242922571;3.17379135683183;3.17603904177395;3.17828548425533;3.18053068448072;3.18277464265635;3.18501735898995;3.18725883369071;3.18949906696932;3.19173805903792;3.19397581011012;3.19621232040103;3.19844759012719;3.20068161950659;3.20291440875870;3.20514595810444;3.20737626776615;3.20960533796765;3.21183316893418;3.21405976089242;3.21628511407047;3.21850922869788;3.22073210500562;3.22295374322608;3.22517414359306;3.22739330634179;3.22961123170890;3.23182791993244;3.23404337125183;3.23625758590794;3.23847056414301;3.24068230620066;3.24289281232593;3.24510208276521;3.24731011776632;3.24951691757841;3.25172248245204;3.25392681263911;3.25612990839293;3.25833176996813;3.26053239762073;3.26273179160809;3.26492995218895;3.26712687962336;3.26932257417275;3.27151703609988;3.27371026566885;3.27590226314509;3.27809302879538;3.28028256288781;3.28247086569180;3.28465793747810;3.28684377851876;3.28902838908718;3.29121176945802;3.29339391990729;3.29557484071229;3.29775453215161;3.29993299450515;3.30211022805410;3.30428623308094;3.30646100986943;3.30863455870462;3.31080687987284;3.31297797366170;3.31514784036006;3.31731648025808;3.31948389364715;3.32165008081996;3.32381504207043;3.32597877769375;3.32814128798635;3.33030257324592;3.33246263377138;3.33462146986291;3.33677908182190;3.33893546995101;3.34109063455409;3.34324457593626];
%     I_GC_c = [37.4996383921856;37.2548332078092;36.4283288124236;36.2171017455181;36.0594526535833;35.9115039243978;35.7206516758486;35.5457687526235;35.4645745008631;35.3593982858726;35.2646446480797;34.5439024389316;34.4496042048382;34.3367536550321;34.2494160725991;34.1552878382091;34.0225318267624;33.8700294479800;33.7322197521813;33.6578192244874;33.5595559196515;33.4289145601177;33.3042268238371;33.1786408925464;33.0235231925436;32.9220890682906;32.7933488016634;32.0956203754132;32.0134149698111;31.9402817946214;31.7996141727154;31.7154953296368;31.6228560667772;31.5414505004883;31.4524481113140;31.3508797425490;31.2854652404785;31.1914306053749;31.1408305534950;31.0959938489474;31.0430858318622;31.0641636481652;30.9690412374643;30.8992959536039;30.8582724791307;30.8052206773024;30.7651990010188;30.7142602480375;30.6480002036461;30.5753599313589;30.5380638562716;30.4429228856013;30.3636076266949;30.2560176849365;30.1547018931462;30.0181754919199;29.8537241862370;29.6561602812547;29.4377786196195;29.1376347175011;28.8615410878108;28.5411135600163;28.1562579961923;28.2315394618897;27.2097022716816;27.2126863964973;26.6591948256764;26.0754377785032;25.4554875012226;24.7569221427885;24.0479588379221;23.2468096224346;22.4772266670018;21.6680129830842;20.8113131927969;19.9163549442443;19.0327712249840;18.0965621637316;17.2038128508614;16.2679657267658;15.3605304774854;14.4543927151244;13.5951173112066;12.6881058999300;11.8622364660335;10.8011194375845;10.2782222618139;9.52750501741191;8.61494293579688;7.93357564852788;7.30044454794664;6.77625944064214;6.16035426579989;5.64497525875385;5.16162362465492;4.72441913531377;4.31454612658574;3.95227755033053;3.59874264093546;3.29018119665293;3.00322911372552;2.74924242496490;2.67832939744743;2.58007078333710;2.49067190473411;2.39487275388869;2.34205549260251;2.25324465762885;2.17310866891045;2.08711825841638;2.01926423902903;1.94911575979623;1.88229705191627;1.82389184767408;1.76412613413988;1.71169094593207;1.65296411902820;1.60233168550444;1.55217055879365;1.50621289274187;1.46380628112730;1.41082620607445;1.37340160284771;1.33974584799371;1.29936150512739;1.26254058638749;1.21817236723899;1.18855787548849;1.15051551711141;1.13178209580990;1.09564126963491;1.05689699918740;1.03223779321436;1.00921412322167;0.982011108924821;0.954196649142988;0.933257621487925;0.910027226521827;0.889860352287722;0.862006058599062;0.838316685025028;0.818082680760775;0.797788343684465;0.782549083868602;0.761265928948320;0.739661568945698;0.730028877798775;0.713922496518267;0.695299867344911;0.678843048156554;0.660838460326802;0.649057366696747;0.629896406621617;0.617109000418718;0.603455210139803;0.589989001697014;0.580678570990164;0.567285610839356;0.557683612799142;0.542622027987168;0.532108400474406;0.518258635461546;0.509293319512140;0.497352793118331;0.488202205124504;0.478828182354006;0.468886500917918;0.460221030314538;0.448104587202362;0.441216649094231;0.432536416075870;0.425109193866992;0.417060076396248;0.408307219708715;0.401393103801021;0.394546945404697;0.385522557899928;0.380612106178446;0.372607525564145;0.366296726081248;0.361762870734754;0.352927542887272;0.347774303544552;0.340362363686539;0.336202140636285;0.327517845542412;0.323161000427133;0.318350108024248;0.312773068088334;0.308253896940966;0.303326886374816;0.298524243296601;0.294257140015375;0.289435833533733;0.284379271618311;0.279920089267782;0.275588732481385;0.272275805408322;0.265975135649143;0.263913217302554;0.260995613065508;0.256283712638261;0.252464453872433;0.249572932713027;0.245678091117762;0.241957725947313;0.239883404681421;0.234639557559059;0.232384547125292;0.227869074445529;0.225798291447451;0.222371988936042;0.219979802092999;0.217518861184104;0.214980917512402;0.211843541970048;0.208961118292305;0.205994100051124;0.203824360208272;0.201058749381979;0.198683961225536;0.196581103304325;0.194255444598214;0.190596526903666;0.188467527386347;0.186598882088540;0.184907665108340;0.182435297445616;0.181351841464676;0.178632855644918;0.177518067175766;0.175030999770603;0.172310029478859;0.170335199522834;0.168928597141419;0.166885872093665;0.165378744039184;0.162626121329917;0.161118313665291;0.160414100848547;0.158590051316222;0.156832935400278;0.156720543960870;0.153592032136451;0.152817084573196;0.151441497989464;0.150090135985959;0.147760548925723;0.146990053306342;0.145922931001558;0.144860247120584;0.142764303904499;0.141755847632909;0.139879437827240;0.139006281832558;0.137331368086629;0.136060935867709;0.135112276901657;0.133737412181438;0.132116486936951;0.131524506276471;0.129908938183876;0.129735945273417;0.128738876495355;0.127101736233621;0.125510445428119;0.124532502369922;0.123586137299227;0.123161959455318;0.121932384227846;0.121726636603451;0.120038483099947;0.119580595712703;0.118836345839512;0.118875274780644;0.117957003927912;0.117231934438890;0.117059080942472;0.115178614399837;0.115408995761404;0.114051913989668;0.112516650678456;0.112578676816154;0.111024072389246;0.110785091511723;0.110345368051266;0.109314517337608;0.108717646038948;0.107061949103043;0.107075749973032;0.106601948598256;0.106306467600103;0.106530862745948;0.104609512936625;0.104019410346268;0.103473019493828;0.102810683308282;0.103122654686751;0.102310852595660;0.101484779161486;0.100574736122695;0.100334514652975;0.100189416951435;0.0994006899845934;0.0995791723669450;0.0979918953345411;0.0980639032940743;0.0981049551962847;0.0968831939397526;0.0967611915226487;0.0963822372721850;0.0962876724305304;0.0952951574008161;0.0950295077721273;0.0947363604415038;0.0948099539292648;0.0935799641716028;0.0932506161361304;0.0930830488988461;0.0932544616728714;0.0922055604856590;0.0914887152837670;0.0910342850529264;0.0906809361576627;0.0905641558872569;0.0899075772522877;0.0894893472328375;0.0895345083260128;0.0890018711739819;0.0886677856440173;0.0884756564678029;0.0879021214587174;0.0871529872155350;0.0876217668441652;0.0870052170433574;0.0869378114765917;0.0864945424420139;0.0867712346628615;0.0863888290073545;0.0865546106716726;0.0861348137265473;0.0855556387669311;0.0852856369758596;0.0858610676070817;0.0854605609491165;0.0850367474199608;0.0858718400808859;0.0854087862542802;0.0844500084810677;0.0843096850621483;0.0842918940182374;0.0844154965458165;0.0841082724311412;0.0840178149145389;0.0845054320758153;0.0833217999731521;0.0834777038132775;0.0827014079554586;0.0835435281260447;0.0847244466143707;0.0843391846805502;0.0841858807160885;0.0836703647209445;0.0841771297185195;0.0836196840839768;0.0833674730752508;0.0831777565275481;0.0840851899758925;0.0835353600864580;0.0833638810049888;0.0832928535653042;0.0835227786250826;0.0835279621628701;0.0831666287817926;0.0837227560029837;0.0836623980193662;0.0834248900506670;0.0835816333432480;0.0831849997677054;0.0827399553001356;0.0830817951283610;0.0842297784965481;0.0833999644874767;0.0835684204459715;0.0836453961781198;0.0834805890932144;0.0835470391183697;0.0834784253131107;0.0834183610223144;0.0841040348641747;0.0836065704251993;0.0839642083432425;0.0840850204633514;0.0850036747498199;0.0842248856254011;0.0843493223349913;0.0840527418343790;0.0842100505171298;0.0849402319950868;0.0843651530985329;0.0850988634565171;0.0853595021218781;0.0858445689602491;0.0854631132262984;0.0860438952815239;0.0859599726910152;0.0869955212390877;0.0860446609448370;0.0863361686601534;0.0867955138459666;0.0873334221515559;0.0870525917108529;0.0873156102189771;0.0874941864329727;0.0882440639358392;0.0885075810624663;0.0885248187521614;0.0891661326475738;0.0889495048148492;0.0893703403051117;0.0895546874486657;0.0902630932048841;0.0903029186428313;0.0907883265204857;0.0906467282229316;0.0911414158888007;0.0917738856341661;0.0916922619185359;0.0915203478873470;0.0922884973034796;0.0926033216712055;0.0931138918545560;0.0935911211306301;0.0936761818857671;0.0944248633828444;0.0948723820317554;0.0949468494003478;0.0954707621579024;0.0961279073472088;0.0962942185501833;0.0962053035211515;0.0966963507939852;0.0979202342844086;0.0980179641064233;0.0983733040196930;0.0986667768940910;0.0997478309271865;0.0998404286449695;0.100449357453439;0.100691676454507;0.101515291174958;0.101793108434166;0.102433177237233;0.103000229279223;0.103531218296803;0.104597478632081;0.104982689214794;0.104778765990754;0.106326827227661;0.106435668737810;0.106973613402831;0.107609565479959;0.108238666449599;0.108694691946741;0.110153897298624;0.110511372533923;0.111018660824552;0.111586810055312;0.112316769366801;0.112870487906306;0.113558568220466;0.114402567317150;0.115342464582022;0.115932153476461;0.117135352061785;0.117500073144297;0.118120015725177;0.119250678028916;0.120246207284863;0.120791227081139;0.122155177744817;0.122882983315590;0.123445015950130;0.124151839333055;0.125352315927278;0.126035099889789;0.126932948080035;0.128461797686879;0.129475140040189;0.130313104545368;0.131493852694764;0.132453620357563;0.133032552880140;0.134300047957459;0.135772027038256;0.136399424971278;0.137788996987622;0.138414776598931;0.139668011592967;0.141375595561552;0.142527124306941;0.143349299990772;0.144551451285932;0.145890295723163;0.147056746963049;0.148365658021746;0.149810704440913;0.151454004226644;0.152793846794285;0.153896830798280;0.155336881296646;0.157197219323716;0.158102403766416;0.160037937048823;0.160855417027481;0.162492032818723;0.163699210517460;0.165387514803600;0.167351960205536;0.168693255472803;0.170764446658187;0.172367763133848;0.174212097864683;0.175663294685604;0.177898421906694;0.179362848900446;0.179249970086033;0.181849881766393;0.183928880268139;0.185997186934555;0.188219542074020;0.190422409526596;0.192784709327254;0.193541064028063;0.195486546591224;0.196686887249801;0.199192614012702;0.200953345393903;0.203610785093689;0.204512692389041;0.206479550973491;0.208478573776161;0.210582263115830;0.212006979202808;0.214664343497077;0.215780240429842;0.218140890258223;0.219480926197594;0.221027251994341;0.223643635887271;0.225810809371972;0.227323362672101;0.229044394284067;0.230754598888442;0.232778602884360;0.234155865246407;0.235993221814910;0.238217985042940;0.239566814189230;0.240515343137583;0.241903344073455;0.244432323559211;0.245866322547838;0.246420910819076;0.248555718878753;0.249884475693517;0.250449999353858;0.251585519201414;0.252779565557130;0.254497722247741;0.255126440084873;0.256349613524879;0.257489971672725;0.257103657393491;0.258312544801924;0.258381990142922;0.259223429176171;0.258758551389667;0.259825997418901;0.260045694679529;0.259891475244714;0.260628482730240;0.260481540795873;0.260504639306441;0.260606652237674;0.260112524477504;0.259923926359401;0.259232156881148;0.259294659466811;0.259191891412954;0.257727247293731;0.257189441657865;0.255892029760549;0.255247284510122;0.254240165882962;0.253651667905458;0.251931132081184;0.250857834404964;0.249122991871621;0.248469565424385;0.246640776838502;0.244169987093193;0.242791967622438;0.240988748408351;0.239654572900907;0.237806385230646;0.235634746220506;0.233529474082008;0.231884595658197;0.229808106535271;0.227120439235811;0.225409909927598;0.222370773577963;0.220274578688395;0.217400590465258;0.215727397482599;0.213149284290822;0.210088118986334;0.207434588941018;0.204573359001446;0.202472437179484;0.199576822762138;0.196714116829165;0.193775002659220;0.191718727974613;0.188278620507074;0.186098140413177;0.182770599430838;0.179771405221572;0.177078675541496;0.174937396784584;0.171554989464862;0.168912232489398;0.165642615444593;0.163284899808782;0.160040840268530;0.157431185588883;0.154487887505419;0.152644031718029;0.149379139483998;0.146642173793427;0.144304869732841;0.141394927354025;0.139183848728402;0.136378238241086;0.133718180962964;0.131522165101923;0.128782999912383;0.126141535362966;0.123990811307635;0.120707778997368;0.118656407995089;0.116558736256450;0.114272405430663;0.111810389716351;0.110110689986860;0.107875433164136;0.105677559858969;0.103758408389972;0.101753851837463;0.0994364280068472;0.0976464360860966;0.0955470808123102;0.0936690528301063;0.0915869018640204;0.0903336018113043;0.0882722203537735;0.0867212214182326;0.0847499737669528;0.0833088709576811;0.0817807817067562;0.0802478752655140;0.0782050168539257;0.0767931118978623;0.0754228620532900;0.0742539905701452;0.0725798106190140;0.0714703242958179;0.0702803735932788;0.0688743990444592;0.0677203687643831;0.0662732155383156;0.0651844245641772;0.0642754403572801;0.0631498280480671;0.0615585494621173;0.0608664059298069;0.0598001541793686;0.0585623921787418;0.0578122164103816;0.0568700084516389;0.0553298267231375;0.0546772685999258;0.0542393927358034;0.0531471401890058;0.0519588455632168;0.0514520098524078;0.0508679152986645;0.0497543442260019;0.0490053758306410;0.0486221850812871;0.0477231729508574;0.0470082652718414;0.0465084943584076;0.0459828679034788;0.0451689947971547;0.0442919128388797;0.0439463498151099;0.0434173195478428;0.0429818054771213;0.0423388355370746;0.0415610036461803;0.0409245784827451;0.0406800346909924;0.0401708683639568;0.0394690011218357;0.0394587909771062;0.0384952715270525;0.0381257551941984;0.0379600738941668;0.0372883034338003;0.0370663930260093;0.0367242473394865;0.0360296001708497;0.0357660313097922;0.0354671924805957;0.0351198186132375;0.0344023914544330;0.0343153616956022;0.0338589697377667;0.0335647783704276;0.0331790324921377;0.0331615113610083;0.0327207980420024;0.0323823954955347;0.0322471477861945;0.0318426733861727;0.0315626544993144;0.0311135326887795;0.0308818165440491;0.0306123852933966;0.0303084081142922;0.0301377397792610;0.0300907380027190;0.0297080529061949;0.0294565130500449;0.0292923134209813;0.0288977724792032;0.0290164359349684;0.0286109546405044;0.0289926109619715;0.0282480306823265;0.0280175586059109;0.0279775505002816;0.0275033393387664;0.0273768434931974;0.0271741309617271;0.0269192380968549;0.0266548784356806;0.0267615185580666;0.0265603315136120;0.0263184626546715;0.0261522597669838;0.0258223278910966;0.0259040816252261;0.0257296228340069;0.0254926620239851;0.0256389227253211;0.0249929291910680;0.0249728931678840;0.0247245787541135;0.0248721362457691;0.0246985039639338;0.0244385338138334;0.0246143215183240;0.0241495595799066;0.0241247720819240;0.0241939643910717;0.0237849599530596;0.0237941704489560;0.0237521197867216;0.0235285289273351;0.0234570331555563;0.0233220870385309;0.0233070372264960;0.0230581193128611;0.0231295022663461;0.0232227228689549;0.0230077753704454;0.0229288119722936;0.0231661757738245;0.0227478748706787;0.0228136579317384;0.0227804463185621;0.0223818087897547;0.0225266154294518;0.0224489831136479;0.0222667554030470;0.0221847844582676;0.0220647200942889;0.0220395032920402;0.0217114821642993;0.0219871901824814;0.0217397871415369;0.0218450921535550;0.0218008450259407;0.0218344714234402;0.0215694654005418;0.0215075150782776;0.0212987985735320;0.0214431131901444;0.0212305136966542;0.0213297237129501;0.0212614495440215;0.0217968742192471;0.0213976187831031;0.0209758211806359;0.0210844154369079;0.0208376402614487;0.0211509225289370;0.0213021876370094;0.0218966399423126;0.0205487396011189;0.0207026795969442;0.0213921673559823;0.0204155125383724;0.0204633379170904;0.0200609684942755;0.0203388414982041;0.0203435677384577;0.0202575859628054;0.0202851951327361;0.0202584230260972;0.0203313586566435;0.0202485470999655;0.0199307003806070;0.0202848816407111;0.0197481404573840;0.0200180144068593;0.0197751220401197;0.0196937051860860;0.0194422718030303;0.0196606155102091;0.0197320803574530;0.0196775592553173;0.0195612658853753;0.0195992321102882;0.0193309335190488;0.0193400583268185;0.0194387430392444;0.0191746906263867;0.0193883106751985;0.0192554895012652;0.0192868113895302;0.0194438283497712;0.0192658445089028;0.0191799866828671;0.0190471864785960;0.0192623669083169;0.0193548560337128;0.0191622056097198;0.0190559010857016;0.0191646345804209;0.0187859801846454;0.0190984067974883;0.0190340818820691;0.0190262681786453;0.0190074757431962;0.0189581788323853;0.0189775105099474;0.0186714493488168;0.0187788258984221;0.0187438133093437;0.0188215813252638;0.0188003652478934;0.0188825051964057;0.0185983047092690;0.0188468176110961;0.0187942831650983;0.0185860248153894;0.0185843402667667;0.0185372378208103;0.0186243592434270;0.0187239700124557;0.0186597144201739;0.0185671743389061;0.0185596309228783;0.0183603842432739;0.0186161755077576;0.0184799585137113;0.0183481104659061;0.0183461182202108;0.0184279628954496;0.0182539557775897;0.0185258838459466;0.0183587881756707;0.0186093728958480;0.0183411159604321;0.0182791254880500;0.0186630761480609;0.0181298667157161;0.0182459612078933;0.0184675831579458;0.0185955748824505;0.0184154570717442;0.0184624266977795;0.0183126033171461;0.0183970370356682;0.0185419906367076;0.0184031608369824;0.0184541739351863;0.0182770065824583;0.0186566948813977;0.0182691447156686;0.0181024413861789;0.0185529050686361;0.0184730534222183;0.0185084257754729;0.0184833263226635;0.0183864933922039;0.0183482356438532;0.0184138270247567;0.0186752125982310;0.0185238592093305;0.0187434838030276;0.0185972783247171;0.0181357392004642;0.0187132227771188;0.0187158945149902;0.0189366971777676;0.0187536555698207;0.0189765925461928;0.0187758429576026;0.0188145984637302;0.0188396037600377;0.0188413862580502;0.0188459430550750;0.0190759678635653;0.0189786901373551;0.0186481443425060;0.0188695988653146;0.0191491672037817;0.0193394446522504;0.0190723177645248;0.0190897099764720;0.0197237878371987;0.0191725315164984;0.0193703298064157;0.0190905799610329;0.0194582912077836;0.0191897423333521;0.0195108815204790;0.0196133443225207;0.0195421135290876;0.0201424324731172;0.0197604328780863;0.0194711128103248;0.0197812065126590;0.0202185478118173;0.0200238140517035;0.0201681282748319;0.0205191728038324;0.0203431656384889;0.0205936516580411;0.0207153992669838;0.0207206707806900;0.0204223851686011;0.0210798794903056;0.0210369952477117;0.0212385673171060;0.0214759869480948;0.0218305289405694;0.0221439840865384;0.0232682859724466;0.0237881365803310;0.0227849939049099;0.0228405695096319;0.0228195279399666;0.0230476486838428;0.0234747644325930;0.0238319906717991;0.0240618057295245;0.0240681163842894;0.0240141494245520;0.0240705409051771;0.0244996976470823;0.0248041127128012;0.0252508323675244;0.0259381278022552;0.0261231091207576;0.0265283422294468;0.0266333082389331;0.0271837099981487;0.0271763761859966;0.0283153637930755;0.0283883946009908;0.0285393812905249;0.0296620117506671;0.0300153022512822;0.0310947767873625;0.0319544052552341;0.0328155829922504;0.0331283560454105;0.0332238582293423;0.0346697362944266;0.0351597622626639;0.0360396379232369;0.0370153248842645;0.0383083139502351;0.0390561120022396;0.0406137456807503;0.0413040506236033;0.0428944285042096;0.0434234956462627;0.0451778494452900;0.0461830896487462;0.0469085254408741;0.0492392655574735;0.0495403179811676;0.0512851486948006;0.0528796363243245;0.0538077672258900;0.0561724637940814;0.0580105609043118;0.0599778835515627;0.0617718673618281;0.0616875458865694;0.0622170402901965;0.0653136053641224;0.0680881850545449;0.0693734423056632;0.0712072811158143;0.0718066821787546;0.0742352895174523;0.0735710941758569;0.0752698185398832;0.0788573342481060;0.0798270625391214;0.0806950489829287;0.0791980500723784;0.0805883508314134;0.0827628891664726;0.0834189421971264;0.0856124722254894;0.0839728949121338;0.0829123442924816;0.0860383899100689;0.0851982209745146;0.0897845006936449;0.0873187129810341;0.0888710435361596;0.0860206979105856;0.0853806243200176;0.0860069437035836;0.0884730603939587;0.0891589554076845;0.0911733861891870;0.0861840852174086;0.0902825163581452;0.0858807373807453;0.0859093474917708;0.0842868756434246;0.0871752511017805;0.0831516318926400;0.0832023998963708;0.0863235308663344;0.0816162923212966;0.0822953877499070;0.0813145085528648;0.0857358144630724;0.0801335035826561;0.0816771050204893;0.0858537247575922;0.0779650141175828;0.0784686978017259;0.0777296491927410;0.0767711441447507;0.0768091386100438;0.0758565155957630;0.0746455690517293;0.0730356907861769;0.0725700185469693;0.0730154488255198;0.0706916252165585;0.0702932891711562;0.0693647807202049;0.0696637556239349;0.0688300976069800;0.0671026054992115;0.0661618440460318;0.0653930304940334;0.0636208596998396;0.0639784034222365;0.0635229980704250;0.0623671078908284;0.0636393741862071;0.0611647215570232;0.0613685831463423;0.0602813250585447;0.0599577571307978;0.0593124415473514;0.0589698604148216;0.0579031515354112;0.0576563734653825;0.0576102970273536;0.0562779062829001;0.0565768260968255;0.0564576134285343;0.0547751054008383;0.0538450533243285;0.0536064668509639;0.0533224619491000;0.0518254662023279;0.0523011630072659;0.0516270561189482;0.0517028170452727;0.0499300632458147;0.0510264266766338;0.0502485827312818;0.0505876790311235;0.0502110242476833;0.0500473147949757;0.0491911407345692;0.0476975090423956;0.0483856879935516;0.0469366149549049;0.0453442993889730;0.0462023027358431;0.0463876562848192;0.0453418296004854;0.0444586257598817;0.0445587267898697;0.0451806253054321;0.0443530800112586;0.0441915944442830;0.0445684958026137;0.0418947171411499;0.0441932722684921;0.0445756131632978;0.0436317802141775;0.0427427454263860;0.0426922927217532;0.0415835282106632;0.0411889052039778;0.0410213188276570;0.0422992472442955;0.0412247369996285;0.0411719078579223;0.0407179272032039;0.0402909378390404;0.0397923907599835;0.0400005482577337;0.0386561450792916;0.0390622401845554;0.0386415696772356;0.0390949596575275;0.0386859596994390;0.0384390111146331;0.0383528156908703;0.0389125312182718;0.0375412660102177;0.0365562411844043;0.0401222106537155;0.0375443465195469;0.0370207350895898;0.0385787683121572;0.0363112885761093;0.0376749688091075;0.0380773846718442;0.0365970300223128;0.0353199987716234;0.0366100764337691;0.0347705940290818;0.0359591666134441;0.0341095252262050;0.0359824512084458;0.0362504606412126;0.0346785880250509;0.0368545668631960;0.0350404429632715;0.0347629248751471;0.0358759237820286;0.0337229423357313;0.0351449217310496;0.0345992732995441;0.0370188029009108;0.0336262016120379;0.0377555442233060;0.0351219932628978;0.0368512970054316;0.0360720304482021;0.0311529300960851;0.0305857332354005;0.0335578236015645;0.0270776921474693;0.0344479515485556;0.0376524121577235;0.0359300145802958;0.0309831285148927;0.0366590293688815;0.0388807887245713;0.0354092897313060];
%    
    A=importdata('./+beamline/glassycarbon_L14_xsection.dat');
    q_GC_c = A(:,1);
    I_GC_c = A(:,2);
    %determine the factor to scale the intensity in absolute scale
    C = [];
    %create a vector to fit the region from q 0.07 to 0.08 A
    %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
    q_range = [q_GC_c(1):0.01:q_GC_m(end-1000)]; %good region for 2 and 7 m flight tube
    %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
    for ii = 1:length(q_range)
        [~, index_m] = min(abs(q_GC_m - q_range(ii)));
        [~, index_c] = min(abs(q_GC_c - q_range(ii)));
        C = [C I_GC_c(index_c)/I_GC_m(index_m)]; %determined in cm/s
    end
    dev_SAXS = std(C);
    C_SAXS = mean(C);
    calibration_factor_SAXS = C_SAXS; %in s/photons
    
    
    if (use_WAXS)
        filename_GC = fullfile(BasePath,'analysis',sprintf('radial_integration_waxs%s',file_flag),sprintf('%s_2_%05d_00000_00000_integ.mat',e_account,Glassy_carbon));
        WAXS_GC = importdata(filename_GC);
        I_GC_WAXS= median(squeeze(WAXS_GC.I_all),2);
        filename_DB = fullfile(BasePath,'analysis',sprintf('radial_integration_waxs%s',file_flag),sprintf('%s_2_%05d_00000_00000_integ.mat',e_account,DB));
        Data_DB_WAXS= importdata(filename_DB);
        I_DB_WAXS = median(squeeze(Data_DB_WAXS.I_all),2);
        q_GC_WAXS = WAXS_GC.q;
        I_GC_WAXS = ((((I_GC_WAXS*scale_GC)*1/transmittance )-(I_DB_WAXS*scale_DB))*1/thickness_L14);
      
        %determine the factor to scale the intensity in absolute scale
        C=[];
        %create a vector to fit the region from q 0.07 to 0.08 A
        %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
        q_range = [q_GC_WAXS(1):0.01:q_GC_WAXS(300) ]; %good region for 2 and 7 m flight tube
        %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
        for i = 1:length(q_range)
            [~, index_m] = min(abs(q_GC_WAXS - q_range(i)));
            [~, index_c] = min(abs(q_GC_c - q_range(i)));
            C = [C I_GC_c(index_c)/I_GC_WAXS(index_m)]; %determined in cm/s
        end
        dev_WAXS = std(C);
        C_WAXS = mean(C);
        calibration_factor_WAXS = C_WAXS; %in s/photons
    else
        calibration_factor_WAXS=1;
        dev_WAXS = 0;
        C_WAXS = 1;
    end
    
    
    %calculate the efficiency of the detector
    % from http://henke.lbl.gov/optical_constants/filter2.html, considering
    %silicon  Si Density=2.33 Thickness=320. microns
    [ val, ~ ] = utils.get_fil_trans( 'Si', [energy,energy+0.01], 320);
    TransmissionSi1 = val(1,2);
    p = 1-(TransmissionSi1);
    
    %contribution from the FT windows to the flux
    % from http://henke.lbl.gov/optical_constants/filter2.html, considering
    % mica KAl3Si3O11.8H1.8F0.2 Density=2.76 Thickness= 7. microns
    [ val, ~ ] = utils.get_fil_trans( 'mica', [energy,energy+0.01], mica_thickness_FT);
    mica = val(1,2);
    
    %from http://henke.lbl.gov/optical_constants/filter2.html, considering
    % mylar C10H8O4 Density=1.43 Thickness=300. microns
    [ val, ~ ] = utils.get_fil_trans( 'mylar', [energy,energy+0.01], 300);
    mylar = val(1,2);
   
    
    % define the length of the air path in mm
    % remove the air path in vacuum from the DetDist_mm
    % values from https://intranet.psi.ch/CSAXS/NewFlightTube
    if DetDist_mm < 5e3
        air_path = DetDist_mm - 2051.5; %for 2m flight tube in mm
    else
        air_path = DetDist_mm - 7030.2; %for 3m flight tube in mm
    end
    
    %calculate the transmission of the air path
    
    %from http://henke.lbl.gov/optical_constants/gastrn2.html considering
    %N1.562O.42C.0003Ar.0094	Pressure=760.	T = 297 K, Path = 1	cm
   % air N1.562O.42C.0003Ar.0094	Pressure=760 torr.	T = 297 K, Path = 1	cm
   [ val,~ ] = utils.get_gas_trans('air', [energy,energy+0.01], air_path/10,760, 297 );
   ap_trans = val(1,2);
   
    %define the solid angle: we consider the first pixel close to the beam stop
    %as it would have the highest intensity
    angle = (atan(pixel_size/DetDist_mm)); %in radians
    sol_angle = angle^2; %in steradian (sr): square radians
    
    %plot the comparison
    figure(1)
    loglog(q_GC_c, I_GC_c,'k', 'LineWidth', 4);
    hold on
    I_GC_m = I_GC_m*calibration_factor_SAXS; % in cm-1
    loglog(q_GC_m, I_GC_m, 'r', 'LineWidth', 2);
    
    if (use_WAXS)
        loglog(q_GC_WAXS, I_GC_WAXS*C_WAXS, 'LineWidth', 2);
    end
    
    xlabel('Scattering vector q (A^{-1})');
    ylabel('Differential scattering cross section (cm^{-1})');
    axis tight
    grid on;
    
    if (use_WAXS)
        legend('Standard GCL14', sprintf('Corrected SAXS GCL14 x %.2d',C_SAXS ), ...
            sprintf('Corrected WAXS GCL14 x %.2d',C_WAXS ), 'Location','southwest')
    else
        legend('Standard GCL14', sprintf('Corrected SAXS GCL14 x %.2d',C_SAXS ),...
            'Location','southwest')
    end
    
    set(gca,'fontsize',12)
    
    %calculate the flux
    positive_cont = (mica * mylar * ap_trans * p);
    flux_GC = 1/(calibration_factor_SAXS*sol_angle * positive_cont); %flux in photons/s
    
    
    %display the results
    fprintf('The approximated flux from GCL14 is %5.2e photons/s \n', flux_GC);
    fprintf('The calibration factor for corrected intensity is C_{SAXS} = %.2d ± %.2d. \n ', calibration_factor_SAXS, dev_SAXS);
    if (use_WAXS)
        fprintf('The calibration factor for corrected intensity is C_{WAXS} = %.2d ± %.2d. \n ', calibration_factor_WAXS, dev_WAXS);
    end
end

%*-----------------------------------------------------------------------*
%|                                                                       |
%|  Except where otherwise noted, this work is licensed under a          |
%|  Creative Commons Attribution-NonCommercial-ShareAlike 4.0            |
%|  International (CC BY-NC-SA 4.0) license.                             |
%|                                                                       |
%|  Copyright (c) 2017 by Paul Scherrer Institute (http://www.psi.ch)    |
%|                                                                       |
%|       Author: CXS group, PSI                                          |
%*-----------------------------------------------------------------------*
% You may use this code with the following provisions:
%
% If the code is fully or partially redistributed, or rewritten in another
%   computing language this notice should be included in the redistribution.
%
% If this code, or subfunctions or parts of it, is used for research in a 
%   publication or if it is fully or partially rewritten for another 
%   computing language the authors and institution should be acknowledged 
%   in written form in the publication: “Data processing was carried out 
%   using the “cSAXS software package” developed by the CXS group,
%   Paul Scherrer Institut, Switzerland.” 
%   Variations on the latter text can be incorporated upon discussion with 
%   the CXS group if needed to more specifically reflect the use of the package 
%   for the published work.
%
% A publication that focuses on describing features, or parameters, that
%    are already existing in the code should be first discussed with the
%    authors.
%   
% This code and subroutines are part of a continuous development, they 
%    are provided “as they are” without guarantees or liability on part
%    of PSI or the authors. It is the user responsibility to ensure its 
%    proper use and the correctness of the results.