% template_tomo_recons.m
% to process ptychographic reconstructions from many projections
% and make tomographic reconstruction

% This code and subroutines are part of a continuous develpment. There is no 
% liability on PSI or cSAXS. Before publishing results using this code 
% please notify M. Guizar-Sicairos, (mguizar@gmail.com)
% or another PSI cSAXS staff member.

% Publications most relevant to this code
% M. Guizar-Sicairos, A. Diaz, M. Holler, M. S. Lucas, A. Menzel, R. A. Wepf, and O. Bunk
% "Phase tomography from x-ray coherent diffractive imaging projections," Opt. Express 19, 21345-21357 (2011).
% For tomographic consistency alignment:
% M. Guizar-Sicairos, J. J. Boon, K. Mader, A. Diaz, A. Menzel, and O. Bunk,
% "Quantitative interior x-ray nanotomography by a hybrid imaging technique," Optica 2, 259-266 (2015).

% prevent unwanted calling of the template 
if length(dbstack) == 1   
     return 
end


close all 

%% Add paths
addpath('utils')
addpath('tests')
base_path='../../';
addpath([base_path 'cxs_software/base/']);
addpath([base_path 'cxs_software/ptycho/']);
addpath([base_path 'cxs_software/ptycho/utils']);
addpath([base_path 'cxs_software/tomo/utils/'])


%matlabpool open

% compatibility with new base structure 
import math.* 
import ptycho.* 
import utils.*
import io.*
import plotting.*

if debug() <= 1  % control point for the automatic code tests

%% File management
%%%%%%%%%%%%%%%%%%%%%%%%%
%%% Edit this section %%%
%%%%%%%%%%%%%%%%%%%%%%%%%

par.scanstomo = [];
par.tomo_id = 139; % Either scan numbers or tomo_id can be given, but not both, if not provided leave tomo_id=[]

% verbosity and online tomo 
par.online_tomo = length(dbstack)==3;   % automatically run if called from externally
par.verbose_level = 1; 
par.base_path=base_path;
par.showrecons = par.verbose_level > 1;

% IO loading 
par.fileprefix='';                  % string at the beginning of the filename, related to reconstruction name
par.filesuffix = '512x512_online_test_1';   % string searched at the end of the data filenames,  No need to add _c or _recons, it will look for it 
par.file_extension = 'h5';
par.downsample_projections = 0;     % downsample projections by factor of 2^x, set 0 to do nothing and 1,2,.. for different levels of projection binning 

par.use_mex_loader = true;   % if true, use the new fast MEX-based loader, false is rather a backup option
par.Nthreads_mexread = 10;   % number of threads used by the fast MEX based data loader
par.analysis_path = [par.base_path 'analysis/'];      % path from where projections are loaded 
par.output_path = [par.base_path 'analysis_tomo/'];   % path where the tomograms are stored 
par.clip_amplitude_quantile = 0.95;  %  clip amplitudes in the loaded projections that are exceeding given quantile, if par.clip_amplitude_quantile == 1, do nothing 
par.max_residua_limit = 100;        % limit used to determine which projection have failed 
par.sort_by_angle = false;          % If true projections are sorted by angle rather than by the scan numbers, this sorting can effect alignment 

% IO saving
par.output_folder_prefix = '';      % a string that will be attached to the output folder name in form tomo_+output_folder_prefix+autogenerated_description
par.save_final_data = true; 
par.force_overwrite = par.online_tomo ;     % ask before overwritting data 
par.save_temp_data = false;      % 1 to save temporal data to disk, ! final results are saved always !

% Angles + auto colibration 
par.use_OMNY_file_angles = 1;        %% Read angles from special file in OMNY, otherwise it uses SPEC dat file
par.checkangles = false;             %% Compares spec angles with expected
par.auto_alignment = 0;     %% Uses interferometer positions to prealign projections - for now it will read them from the spec file. 
			% This feature needs a new calibration of the sphere and is not fully tested.
par.get_auto_calibration = 1;  % If you want to use this recons to get a calibration file for autotomo  
par.remove_duplicated_angles = true; % remove angles measured twice, should be true for FBP method 
par.surface_calib_file = [par.base_path 'matlab/tomo/position_calibration_20181026.mat'];
par.omnyposfile = [par.base_path 'specES1/scan_positions/scan_%05d.dat'];		%Filename pattern for Orchestra interferometer position files   
par.OMNY_angle_file = [par.base_path, 'specES1/dat-files/tomography_scannumbers.txt'];  % Filename with angles

% Other
par.showsorted = true;      % sort plotted projections by angles, it has not effect on the alignment itself 
par.windowautopos = true;
par.save_memory = false;        % try to limit use of RAM 
    par.inplace_processing = par.save_memory; % process object_stack using inplace operations to save memory 
    par.fp16_precision     = par.save_memory; % use 16-bit precision to store the complex-valued projections 


par.GPU_list = [1];     % number of the used GPU % If you want to check usage of GPU 
                        % > nvidia-smi
                        % Then in matlab use  par.GPU_list = 2  for example to use the second GPU 
par.Nworkers = min(10,feature('numcores'));  % number of workers for parfor. avoid starting too many workers

% Geometry settings, important for laminography 
par.is_laminography = false;        % false -> standard tomography, true = allow some specific options for laminography reconstruction, ie circular field of view ... 

end
if debug() == 1 % control point for the automatic code tests
    return
end

if debug() == 0 % ignore during automatic tests 

%%% load angles for the projections 
[par,theta] = prepare.load_angles(par, par.scanstomo, par.tomo_id);



%% Reads ptychographic reconstructions and stores them in stack_object
%%%%%%%%%%%%%%%%%%%%%%%%%
%%% Edit this section %%%
%%%%%%%%%%%%%%%%%%%%%%%%%
exclude_scans = [];
%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%


%%% perform initial checks 
[par, angles_check, object] = prepare.initialize_tomo(par, par.scanstomo, true); 


[stack_object, theta, num_proj, par] = prepare.load_projections_fast(par,exclude_scans,par.dims_ob,theta,object_preprocess_fun);

end

% get some very preliminary phase ramp removal 
utils.verbose(-1,'Stabilize phase ramp')
stack_object = tomo.block_fun(@utils.stabilize_phase,stack_object, 'weight', par.illum_sum / max(par.illum_sum(:)));



%% Tweak automask parameters - Find good settings for automatic detection of air
%%%%%%%%%%%%%%%%%%%%%%%%%
%%% Edit this section %%%
%%%%%%%%%%%%%%%%%%%%%%%%%
whichprojection = 250;% Select which projection to try
smoothing = 20;     % Size of smoothing window
gradientrange = 2;  % Range of the phase gradient taken. In units of radians per pixel
close_size = 10;%5;   % Size of closing window, removes dark bubbles from the mask ( = 1 for no effect)
open_size = 50;    % Size of opening window, removes bright bubbles from mask ( = 1 for no effect)
maskzero_columnrange = [];  % Columns to be forced so the mask is zero. e.g. = [200:500], could be useful for middle of cylinders

show_bivariate = par.verbose_level > 3;     % Show the gradient bivariate histogram, input figure here
show_object =  1; 
%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%


mask1 = utils.auto_mask_find(object,'margin',par.asize/2,'smoothing',smoothing,...
    'gradientrange',gradientrange,'close_size',close_size,'open_size',open_size,...
    'show_bivariate',show_bivariate,'zero_columns',maskzero_columnrange);

if show_object && ~debug()
    scan_0 = par.scanstomo(1);   % Choose scan number to be used as an example                        
    ptycho_filename = find_ptycho_filename(par.analysis_path,scan_0,par.fileprefix,par.filesuffix, par.file_extension);

    plotting.ptycho_show_recons(ptycho_filename);

    figure(6);
    subplot(2,2,1);
    imagesc(angle(object).*mask1);
    axis xy equal tight
    colormap bone
    title('auto mask');

    remove_ramp = 1;
    if remove_ramp
        display(['Removing phase ramp'])
        [object ph_err] = utils.remove_linear_phase_smart(object,'mask',mask1);
    end

    subplot(2,2,2);
    imagesc(angle(object));
    axis xy equal tight
    colormap bone
    title(sprintf('Ramp corrected projection %d (scan %d)',whichprojection, par.scanstomo(whichprojection)));

    subplot(2,2,[3 4]);
    line_mid = floor(size(object,1)/2);
    plot(angle(object(line_mid,:))); grid on;
    title(sprintf('Line %d profile',line_mid))
end

% for xx = 1  
%% load angles for the projections 


% %% Check and plot angles
% scans = scan_orig;
% Nscans = length(scans);
% angles = nan(Nscans,1);
% 
% 
% if ~par.use_OMNY_file_angles
%     S = spec_read(par.base_path,'ScanNr',scans);
%     for ii = 1:Nscans
%         angles(ii)=S{ii}.samroy;
%     end
% else
%     [S, errflag] = beamline.read_omny_angles(par.OMNY_angle_file,scans);
%     if errflag
%         display(['Not all scans found in ' par.OMNY_angle_file])
%         display(['I will remove the angles not found and show you some plots anyway'])
%         scans = S.scan(:).';
%     end
%     angles=S.readout_angle(:).';
% end
% 
% ind = find(angles<179.99);
% theta = angles(ind);    % Angles not repeated in scan
% scanstomo = scans(ind);
% num_proj=numel(scanstomo);
% [thetasort,indsortangle] = sort(theta);
% 
% if par.sort_by_angle
%     theta = theta(indsortangle);
%     scanstomo = scanstomo(indsortangle);
% end
% 
% figure(1);
% subplot(2,1,1)
% plot(scanstomo,theta,'ob'); grid on;
% xlabel('S#');
% legend('Spec angles')
% subplot(2,1,2)
% plot(diff(thetasort))
% title('Angular spacing'); grid on;
% set(gcf,'Outerposition',[139 163 946 815])    %[left, bottom, width, height
% drawnow 
% filename = fullfile(par.output_folder,sprintf('%s_angles.png',p.run_name));
% fprintf('saving %s\n',filename);
% print('-dpng','-r300',filename);


online_tomo_fn_out = sprintf('online_tomo_S%05d_',par.scanstomo(1));



% Choose reconstructed region
%%%%%%%%%%%%%%%%%%%%%%%%%
%%% Edit this section %%%
%%%%%%%%%%%%%%%%%%%%%%%%%
[Nx,Ny,Nangles] = size(stack_object);
object_ROI = {ceil(par.asize(1)/2:Nx-par.asize(1)/2),ceil(par.asize(2)/2:Ny-par.asize(2)/2)}; 
%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%

%% Autotomo - Position correction with sphere calibration (good to 100 nm)
%%%%%%%%%%%%%%%%%%%%%%%%%
%%% Edit this section %%%
%%%%%%%%%%%%%%%%%%%%%%%%%
% If auto_alignment = 1 then it will use the calibration file
% If get_auto_calibration = 1 then it will write in this file the ouput of calibration
omnyposfile = [par.base_path 'specES1/scan_positions/scan_%05d.dat'];		%Filename pattern for Orchestra interferometer position files   
%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%
%%% To improve: Shifts of the probe are not yet considered here, see
%%% /cSAXS_sxdm_2013_06_omny/matlab/tomo/autotomo_calibration_porous_S00506_S00930.m
if ~debug()
[delta_stack_prealign, obj_interf_pos_x, obj_interf_pos_y] = prepare.get_auto_tomo(par,par.surface_calib_file, par.omnyposfile, theta, par.scanstomo);
end

%% prepare parpool 
pool = gcp('nocreate'); 
if isempty(pool) || pool.NumWorkers < par.Nworkers
    delete(pool);
    pool = parpool(par.Nworkers);
end
pool.IdleTimeout = 600; % set idle timeout to 10 hours

%% [**] Compute mask and attempt to remove the phase ramp
%%%%%%%%%%%%%%%%%%%%%%%%%
%%% Edit this section %%%
%%%%%%%%%%%%%%%%%%%%%%%%%
force_remove_ramp = false;
mask_auto_save = true;
show_mask = par.verbose_level > 0 ; 
%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%
tic
num_proj = size(stack_object,3);
failed_ramp = zeros(num_proj,1);
ob_size = size(stack_object(:,:,1));
ph_std = failed_ramp;
mask = false(size(stack_object));

poolobj = gcp('nocreate');
if isempty(poolobj) || poolobj.NumWorkers < par.Nworkers
    delete(poolobj);
    parpool(par.Nworkers);
end
           
parfor ii = 1:num_proj
%     file = find_ptycho_filename(par.base_path, par.scanstomo(ii), par.fileprefix, par.filesuffix, par.file_extension);
%     if iscell(file)
%         warning('Returned file is a cell, using file{1}. Better have specific file name.');
%         file = file{1}; 
%     end
%     
%     if ~io.HDF.hdf5_dset_exists(file,'object_ramp_removed', '/reconstruction') || force_remove_ramp
        display(['Finding mask for projection: ' num2str(ii)])
        object = stack_object(:,:,ii);
        mask1 = utils.auto_mask_find(object,'margin',par.asize/2,'smoothing',smoothing,...
            'gradientrange',gradientrange,'close_size',close_size,'open_size',open_size,...
            'show_bivariate',show_bivariate,'zero_columns',maskzero_columnrange);

        display(['Removing phase ramp for projection: ' num2str(ii)])
        [object_ramp_removed ph_err] = utils.remove_linear_phase_smart(object,'mask',mask1);

%         display(['  Append mask1 and object_ramp_removed to file ' file])
        
        % save data to reconstruction file 
%         try
%             io.HDF.save2hdf5(file, mask1, 'data_name', 'mask1', 'gpath', 'reconstruction', 'extendable', true);
%             io.HDF.save2hdf5(file, object_ramp_removed, 'data_name', 'object_ramp_removed', 'gpath', 'reconstruction', 'extendable', true);
%             io.HDF.save2hdf5(file, ph_err, 'data_name', 'ph_err', 'gpath', 'reconstruction', 'extendable', true);
%         catch
%             display(['  !! Append failed for ' file])
%         end
%     else
%         display(['  Load mask1, object_ramp_removed from ' file])
%         mask1 = io.HDF.hdf5_load(file, '/reconstruction/mask1', '-c');
%         object_ramp_removed = io.HDF.hdf5_load(file, '/reconstruction/object_ramp_removed', '-c');
%         ph_err = io.HDF.hdf5_load(file, '/reconstruction/ph_err', '-c');
%     end
    mask(:,:,ii) = mask1;
    stack_object(:,:,ii) = object_ramp_removed;
    ph_std(ii) = ph_err; 

    if ~any(mask1(:) == 1)
        failed_ramp(ii) = 1;
    end
end


failed_ramp_ind = find(failed_ramp==1);
if ~isempty(failed_ramp_ind)
    warning(['I could not find a mask, for these projections you''ll have to fix the parameters or fix it in the GUI: ' num2str(failed_ramp_ind')])
end

if show_mask
    figure(1000)
    subplot(2,1,1)
    plot(squeeze(max(max(mask,[],1),[],3)),'o')
    title('Horizontal mask profiles')
    subplot(2,1,2)
    plot(ph_std)
    title('Standard deviation of phase within the mask')
    drawnow
end

toc

%% [:D] Save mask
if mask_auto_save && (~par.online_tomo) && ~debug()
    utils.savefast_safe([par.output_folder,sprintf('mask_auto_S%05d.mat',scans(1))], 'mask')
end

%% [:)] Save image stack
%%%%%%%%%%%%%%%%%%%%%%%%%
%%% Edit this section %%%
%%%%%%%%%%%%%%%%%%%%%%%%%
savedata = par.save_temp_data;
%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%
if (savedata || par.save_memory) && ~par.online_tomo && ~debug()
    utils.savefast_safe([par.output_folder 'reconstructed_projections.mat'],'par','stack_object','scans', 'theta','num_proj');
end

%% Display reconstructed phases
%%%%%%%%%%%%%%%%%%%%%%%%%
%%% Edit this section %%%
%%%%%%%%%%%%%%%%%%%%%%%%%
show_phases = par.verbose_level > 0 ; 
show_mask_overlay = true;
par.fps = 4;
par.baraxis = 'auto';       % = 'auto'  or   = [-1 1]
par.showsorted = 0;      % sort the projections by angles 
par.windowautopos = true;
%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%
if show_mask_overlay
    showwhat = stack_object.*exp(1i*mask);
else
    showwhat = stack_object;
end

if show_phases
    tomo.show_projections(showwhat, theta, par, 'fnct', @angle, 'title', 'Phase ramp removed phase', 'plot_residua', true)
end
clear showwhat

%% Choose mask for phase ramp
%%%%%%%%%%%%%%%%%%%%%%%%%
%%% Edit this section %%%
%%%%%%%%%%%%%%%%%%%%%%%%%
% For parallel computation you can start here   parpool(16)
useGUI = 0;
%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%
if useGUI
    GUIhandle = findmask(stack_object,mask);
    datahandles = guidata(GUIhandle);
    stack_phase_corr = angle(datahandles.stack_object);
    mask = datahandles.mask_ramp;
    close(GUIhandle)
end

stack_phase_corr = angle(stack_object);
    

%% Display ramp corrected phases
%%%%%%%%%%%%%%%%%%%%%%%%%
%%% Edit this section %%%
%%%%%%%%%%%%%%%%%%%%%%%%%
par.fps = 10;
par.baraxis = 'auto';       % = 'auto'  or   = [-1 1]
par.showsorted = 0;      % sort the projections by angles 
par.makemovie = 0;  
par.out_fn = 'ramp_corrected'; % for movie output
%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%
if show_phases
    tomo.show_projections(stack_phase_corr, theta, par, 'title', 'stack_phase', 'plot_residua', true)
end

%% [:)] Save ramp corrected phases (stack_phase_corr)
%%%%%%%%%%%%%%%%%%%%%%%%%
%%% Edit this section %%%
%%%%%%%%%%%%%%%%%%%%%%%%%
savedata = par.save_temp_data;
%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%
% If you saved before you may clear stack_object
tic
if savedata && ~debug()
    utils.savefast_safe([par.output_folder 'ramp_corrected_projections.mat'], 'stack_phase_corr','mask','scans','par','theta','num_proj');
end
toc
if par.save_memory 
    clear stack_object
end

%% Find residues
display('Calculating residues')
% Find and display map of residues
residues = abs(utils.findresidues(stack_phase_corr)) > 0.1;
worst_frame = math.argmax(math.sum2(residues(object_ROI{:},:)));
residues_perframe = residues;
residues = sum(residues,3);
figure(10); clf
imagesc(residues); grid on;
set(gcf,'Outerposition',[41 40 1210 967]);
axis xy tight; colormap jet
caxis([-1 1]); colorbar
if ~debug()
filename = fullfile(par.output_folder,sprintf('%s_residues.png',p.run_name));
fprintf('saving %s\n',filename);
print('-dpng','-r300',filename);
end

%% Choose parameters and well-behaved area for phas174e unwrapping, + plot residues %%%%%%%%%%
% No residues is very fast
% Lots of residues is madness and can fail
% This needs to be larger than the area you like to align afterwards%%%%%%%%%%%%%%%%%%%%%%%%%
%%% Edit this section %%%
%%%%%%%%%%%%%%%%%%%%%%%%%
show_residua = par.verbose_level > 0;
deltax = par.asize(2)/2;           % From edge of region to edge of image in x
ry  = [par.asize(1)/2:size(stack_phase_corr,1)-par.asize(1)/2];        % Range in y
pixel=[deltax+5,ry(end)-5];       % Pixel in air (x,y), make sure its air in ALL projections
frames = worst_frame;          % empty = all 

par.baraxis     = 'auto';       % = 'auto'  or   = [-1 1]
par.showsorted  = false;      % sort the projections by angles
par.makemovie   = false;  
%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%
rx=[1+deltax:size(stack_phase_corr,2)-deltax];

if show_residua
    figure(2)
    plot(squeeze(sum(sum(residues_perframe(par.asize(1)/2:end-par.asize(1)/2,par.asize(2)/2:end-par.asize(2)/2,:),1),2)))
    axis tight
    title('Number of residues per projection')

    figure(1)
    tomo.show_projections(stack_phase_corr, theta, par, 'title', 'Phase residua', 'rectangle_pos', [rx([1,end]), ry([1,end])], 'init_frame', worst_frame,  'plot_residua', false)
    hold on
    [i,j] = find(any(residues,3));
    plot(j,i, 'o');
    plot(pixel(1),pixel(2),'or')
    hold off

    display(['Chosen region contains ' num2str(round(math.sum2(residues(ry,rx)))) ' residues in total'])
    if (pixel(1)<rx(1))||(pixel(1)>rx(end))||(pixel(2)<ry(1))||(pixel(2)>ry(end))
        error('Pixel of air is outside of region of unwrapping')
    end
end

%% [**] Phase unwraping on well-behaved area
% For normal operation use a for loop
% For parallel operation start matlabpool(8) and use a parfor, use
% matlabpool close when done
%%% Edit this section %%%
%%%%%%%%%%%%%%%%%%%%%%%%%
force_unwrap = 0;
show_unwrapped_phase = par.verbose_level > 1; 
%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%

tic
%figure(1); clf
%%%%%%%%%%%%%%%%%%%%%%%%
% Do the following for parfor when working on the DaaS cluster:
% poolobj = gcp('nocreate');
% if isempty(poolobj) || poolobj.NumWorkers < par.Nworkers
%     delete(poolobj);
%     parpool(par.Nworkers);
% end
%%%%%%%%%%%%%%%%%%%%%%%%%%
stack_unwrap = zeros(size(stack_phase_corr), 'single');
parfor ii=1:num_proj
    display(['Unwrapping projection ' num2str(ii)])
    this_img=squeeze(stack_phase_corr(:,:,ii));
    
%     file = find_ptycho_filename(par.base_path, par.scanstomo(ii), par.fileprefix, par.filesuffix, par.file_extension);
%     if iscell(file)
%         warning('Returned file is a cell, using file{1}. Better have specific file name.');
%         file = file{1};
%     end   

%     if ~io.HDF.hdf5_dset_exists(file,'object_phase_unwrap', '/reconstruction') || force_unwrap
        sel = this_img(ry,rx);
        %sel_unwrap=unwrap(sel,[],2);  % Matlab's routine, fast but problematic
        %with noise
        tic
        sel_unwrap = utils.goldsteinunwrap2(sel,0);  % Goldstein's method, very robust (gets slow if there are residues)
        toc
        object_phase_unwrap = this_img;
        object_phase_unwrap(ry,rx) = sel_unwrap;
        object_phase_unwrap(ry,rx) = object_phase_unwrap(ry,rx) - 2*pi*round(object_phase_unwrap(pixel(2),pixel(1))/(2*pi));
        
%         display(['  Append unwrapping image to file ' file])              
%         try
%             io.HDF.save2hdf5(file, object_phase_unwrap, 'data_name', 'object_phase_unwrap', 'gpath', 'reconstruction', 'extendable', true);
%         catch
%             display(['  !! Append failed for' file])         
%         end
%     else
%         display(['  Load unwrapped image from ' file])
%         object_phase_unwrap = io.HDF.hdf5_load(file, '/reconstruction/object_phase_unwrap', '-c');
%     end
    
    stack_unwrap(:,:,ii) = object_phase_unwrap;
end

toc

%% Display unwrapped phases
%%%%%%%%%%%%%%%%%%%%%%%%%
%%% Edit this section %%%
%%%%%%%%%%%%%%%%%%%%%%%%%

par.baraxis = 'auto';       % = 'auto'  or   = [-1 1]
par.showsorted = 0;      % sort the projections by angles
par.makemovie = 0;  
par.out_fn = 'unwrapped'; % for movie output
par.fps=3;
%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%
xi=rx(1);
xf=rx(end);
yi=ry(1);
yf=ry(end);
if show_unwrapped_phase
    tomo.show_projections(stack_unwrap, theta, par, 'fps',par.fps, 'title', 'Unwrapped phase', 'rectangle_pos', [xi,xf,yi,yf], 'plot_residua', true)
end


% 
% 
% %% Getting rid of bad projections %%%
% %%%%%%%%%%%%%%%%%%%%%%%%%
% %%% Edit this section %%%
% %%%%%%%%%%%%%%%%%%%%%%%%%
% whichempty2=[]; 
% whichexist2=[];
% %%%%%%%%%%%%%%%%%%%%%%%%%
% %%%%%%%%%%%%%%%%%%%%%%%%%
% 
% tt=0;
% for hh=1:num_proj
%     if min(abs(whichempty2-hh)) ~= 0
%         tt=tt+1;
%         [dummy,ind]=min(abs(whichempty2-hh));
%         whichexist2(tt)=hh;
%     end
% end
% %
% if ~isempty(whichempty2)
%         userans = input(['Do you want remove bad projections and keep going (y/n)? '],'s');
%         if strcmp(lower(userans),'y')
%             display('Removing bad projections. stack_unwrap, scanstomo, theta and num_proj are modified')
%             stack_unwrap = stack_unwrap(:,:,whichexist2);
%             scanstomo = scanstomo(whichexist2);
%             theta = theta(whichexist2);
%             num_proj=numel(scanstomo);
%         else
%             display('Keeping empty spaces for missing projections. Problems are expected if you continue.')
%         end
% end

%% [:)] Save unwrapped phases (stack_unwrap)
%%%%%%%%%%%%%%%%%%%%%%%%%
%%% Edit this section %%%
%%%%%%%%%%%%%%%%%%%%%%%%%
savedata = par.save_temp_data;
%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%
tic
% If you saved before you may clear stack_phase_corr
if savedata && ~debug()
    utils.savefast_safe([par.output_folder 'stack_unwrap_projections.mat'], 'stack_unwrap','scans','par', 'theta','num_proj','rx','ry');
end
toc
if par.save_memory
   clear stack_phase_corr 
end

%% Choose region for alignment
%%%%%%%%%%%%%%%%%%%%%%%%%
%%% Edit this section %%%
%%%%%%%%%%%%%%%%%%%%%%%%%
deltaxal = deltax+10;       % From edge of region to edge of image in x
limsy  = [ry(1)+50 ry(end)-50];      % Window inside regstack used for alignment delta = 200; % Window should be centered in x to ensure compliance with iradonfast limsx = [1+delta size(regstack,2)-delta];


par.baraxis = 'auto';       % = 'auto'  or   = [-1 1]
par.showsorted = 0;      % sort the projections by angles 
par.makemovie = 0;  
%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%
limsx=[1+deltaxal size(stack_unwrap,2)-deltaxal];%THIS HAS TO BE CENTERED!!!
if (limsy(1)<ry(1))||(limsy(2)>ry(end))
    warning(['Alignment area is outside than unwrapping area in y'])
end
if (limsx(1)<rx(1))||(limsx(2)>rx(end))
    warning(['Alignment area is outside than unwrapping area in x'])
end
tomo.show_projections(stack_unwrap, theta, par, 'title', 'Unwrapped phase', 'rectangle_pos',[limsx, limsy])


%% Register (align) projections by vertical mass fluctuations and center of mass
%%%%%%%%%%%%%%%%%%%%%%%%%
%%% Edit this section %%%
%%%%%%%%%%%%%%%%%%%%%%%%%;
disp = 2;               % = 0 no display, =1 only final display, >1 every iteration
pixtol = 0.2;             % Tolerance of registration in pixels
rembias = true;         % Remove bias for y registration
maxorder = 1;           % Max order of bias to remove
params.expshift = true; % Shift in phasor space at the end
params.interpmeth = 'sinc';  % 'sinc' or 'linear' better for noise
params.alignx = false;  % Align horizontally with center of mass
%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%
tic

deltastack = zeros(2,size(stack_unwrap,3), 'single');
if par.auto_alignment && any(delta_stack_prealign(:))
     deltastack = delta_stack_prealign;
end

%%%
%%% Here you can add an initial guess to the positions
%%%
figure(200); clf
set(gcf,'Outerposition',[1 521 808 504]);
figure(1); clf
set(gcf,'Outerposition',[809 521 472 504]);
figure(201); clf
set(gcf,'Outerposition',[1 48 808 473]);
figure(2); clf
set(gcf,'Outerposition',[809 48 472 473]);
[deltastack,aligned] = tomo.alignprojections_v4(stack_unwrap,limsy,limsx,deltastack,pixtol,rembias,maxorder,disp,params);

if ~debug()
figure(200);
filename = fullfile(subdir_tomo,sprintf('%sy_align_1.png',online_tomo_fn_out));
fprintf('saving %s\n',filename);
print('-dpng','-r300',filename);
figure(1);
filename = fullfile(subdir_tomo,sprintf('%sy_align_2.png',online_tomo_fn_out));
fprintf('saving %s\n',filename);
print('-dpng','-r300',filename);
figure(201);
filename = fullfile(subdir_tomo,sprintf('%sy_align_3.png',online_tomo_fn_out));
fprintf('saving %s\n',filename);
print('-dpng','-r300',filename);
figure(2);
filename = fullfile(subdir_tomo,sprintf('%sy_align_4.png',online_tomo_fn_out));
fprintf('saving %s\n',filename);
print('-dpng','-r300',filename);
end

%% Here you can change parameters and do a refinement alignment
%%% Example:
%%% params.alignx = true;  % Align horizontally with center of mass
%limsy  = [268 330];        % Window inside regstack used for alignment delta = 200; % Window should be centered in x to ensure compliance with iradonfast limsx = [1+delta size(regstack,2)-delta];
%deltastack(1,:) = conv(deltastack(1,:),ones(1,3)/3,'same'); % averaging that helps if a single projection jumps out
maxorder = 2;
[deltastack] = tomo.alignprojections_v4(stack_unwrap,limsy,limsx,deltastack,pixtol,rembias,maxorder,disp,params);
%%
maxorder = 3;
[deltastack] = tomo.alignprojections_v4(stack_unwrap,limsy,limsx,deltastack,pixtol,rembias,maxorder,disp,params);
%%
maxorder = 5;
[deltastack,aligned] = tomo.alignprojections_v4(stack_unwrap,limsy,limsx,deltastack,pixtol,rembias,maxorder,disp,params);
% Make sure that deltastack contains all the alignment displacements

% Here one could open already processed data
% load([par.output_folder 'aligned_phases.mat'])

toc
%% jpegs of alignment
if ~debug()
figure(200);
filename = fullfile(subdir_tomo,sprintf('%sy_align_1b.png',online_tomo_fn_out));
fprintf('saving %s\n',filename);
print('-dpng','-r300',filename);
figure(201);
filename = fullfile(subdir_tomo,sprintf('%sy_align_3b.png',online_tomo_fn_out));
fprintf('saving %s\n',filename);
print('-dpng','-r300',filename);
figure(2);
filename = fullfile(subdir_tomo,sprintf('%sy_align_4b.png',online_tomo_fn_out));
fprintf('saving %s\n',filename);
print('-dpng','-r300',filename);
end

%% Show movie with aligned images
%%%%%%%%%%%%%%%%%%%%%%%%%
%%% Edit this section %%%
%%%%%%%%%%%%%%%%%%%%%%%%%
show_aligned_projections = (par.verbose_level > 0); 

par.baraxis = 'auto';       % = 'auto'  or   = [-1 1]
par.showsorted = 0;      % sort the projections by angles 
par.makemovie = 0;  
par.movie_filename = [par.output_folder 'vertical_aligned.avi']; % for movie output
par.movie_range = [1:20];
par.fps=4;
%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%
if show_aligned_projections
    tomo.show_projections(aligned, theta, par,'fps',par.fps)    
    if par.makemovie
        if isempty(par.movie_range)
            par.movie_range = 1:size(aligned,3);
        end
        save_movie(aligned(:,:,par.movie_range), par.movie_filename,'bone')
    end
end

display(['Recommended minimum crop, y = [' num2str(-min(deltastack(1,:))) ',' num2str(size(aligned,1)-max(deltastack(1,:))) '], deltax = ' num2str( max(abs(deltastack(2,:))))])

%% Show radiation damage with vertical derivative phasors
regionx = [par.asize(2)/2:size(aligned,2)-par.asize(2)/2];
regiony = [par.asize(1)/2:size(aligned,1)-par.asize(1)/2];

deraligned = angle(exp(1i*aligned(2:end,:,:)).*exp(-1i*aligned(1:end-1,:,:)));
massdery = squeeze(sum(deraligned(regiony,regionx,:),2));
%
figure(1)
plotting.imagesc3D(deraligned)

fig5 = figure(5);
subplot(3,1,1)
imagesc(massdery)
axis tight xy
colormap plotting.franzmap
caxis([-10 60])
title(sprintf('Vertical mass derivative S%05d - S%05d',par.scanstomo(1),par.scanstomo(end)))
xlabel('Projection number')

subplot(3,1,2)
% imagesc(conv2(massdery-median(massdery,2),[1 1 1 1]/4,'same'))
mass_filt = medfilt1(massdery-median(massdery,2),15,[],2); 
imagesc(mass_filt)
axis tight xy
colormap bone
caxis([-10 10])
title('Filtered change with respect to median')
xlabel('Projection number')

subplot(3,1,3)
% plot(mass_filt(plotrange,:)')
% title(sprintf('Filtered change with respect to median, lines %d - %d',plotrange(1),plotrange(end)))
% plotrange = [30:40];
Nmodes = 2; 
[U,S,V] = svd(medfilt1(massdery,15,'truncate',2)); 
plot( V(:,[2:Nmodes+1]) * S(2:Nmodes+1,2:Nmodes+1) )
title('PCA decomposition - shows changes in the sample')
axis tight
xlabel('Approx projection number')

if ~debug()
file_jpeg = fullfile(par.output_folder,'Radiation_damage_curve.png');
display(['Saving ' file_jpeg])
print(fig5,'-dpng',file_jpeg)
end
%% Clear memory 
if par.save_memory
    clear stack_unwrap 
end

%% Align refinement using derivative in y - not sure if it is useful for general samples
% limsxder = [1 size(deraligned,2)];
% limsyder = [400 550];
% deltastack_dery = deltastack*0;
% maxorder = 1;
% 
% % [deltastack_dery] = tomo.alignprojections_v4(deraligned,limsyder,limsxder,deltastack_dery,pixtol,rembias,maxorder,disp,params);
% %% Shift projections again
% %%%%%%%%%%%%%%%%%%%%%%%%%
% %%% Edit this section %%%
% %%%%%%%%%%%%%%%%%%%%%%%%%
% % interpmeth = 'sinc';          % Interpolation method for subpixel shift
% % %%%%%%%%%%%%%%%%%%%%%%%%%
% % %%%%%%%%%%%%%%%%%%%%%%%%%
% aligned_2 = angle(imshift_fft(exp(1i*aligned),0,-deltastack_dery(1,:)));

%% Choose region for tomography
%%%%%%%%%%%%%%%%%%%%%%%%%
%%% Edit this section %%%
%%%%%%%%%%%%%%%%%%%%%%%%%
show_tomo_region = (par.verbose_level > 0); 
valx = deltax;              % Distance from edge of window
roiy = max(1,min(size(aligned,1),[par.asize(1)/2-50 size(aligned,1)-par.asize(1)/2+50]));        % Max and min y

par.baraxis = 'auto';       % = 'auto'  or   = [-1 1]
par.showsorted = 0;      % sort the projections by angles
 
%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%
roix=[1+valx size(aligned,2)-valx];
if show_tomo_region
    tomo.show_projections(aligned, theta, par, 'rectangle_pos',[roix, roiy])
end

%% Crop the ROI, compute derivative, and show the result
%%%%%%%%%%%%%%%%%%%%%%%%%
%%% Edit this section %%%
%%%%%%%%%%%%%%%%%%%%%%%%%
cut=5;  % remove edge regions 
%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%

% Compute derivative
cut=5;
aligned_diff = math.get_phase_gradient_1D(aligned(roiy(1):roiy(2),roix(1):roix(2),:),2);
aligned_diff = aligned_diff(:,1+cut:end-cut,:);



%% See derivatives
%%%%%%%%%%%%%%%%%%%%%%%%%
%%% Edit this section %%%
%%%%%%%%%%%%%%%%%%%%%%%%%
show_phase_derivatives = (par.verbose_level > 0); 
par.baraxis = 'auto';       % = 'auto'  or   = [-1 1]
par.showsorted = 0;      % sort the projections by angles 
par.makemovie = 0;
par.movie_filename = [par.output_folder 'vertical_aligned_derivatives.avi']; % for movie output
par.movie_range = [1:100];
par.fps=4;
%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%
if show_phase_derivatives
    tomo.show_projections(aligned_diff, theta, par,'fps',par.fps)
    if par.makemovie
        if isempty(par.movie_range)
            par.movie_range = 1:size(aligned,3);
        end
        save_movie(aligned_diff(:,:,par.movie_range), par.movie_filename,'bone')
    end
end

%% Clear memory 
if par.save_memory
    clear aligned 
end

%% Visualize some sinograms and tomograms
%%%%%%%%%%%%%%%%%%%%%%%%%
%%% Edit this section %%%
%%%%%%%%%%%%%%%%%%%%%%%%%
slice_num = round(size(aligned_diff,1)/2);        % Choose the slice
filter_type = 'Hann';   % Filter to use for FBP
freq_scale = 1;       % Frequency cutoff
baraxis = [-0.5 0.5];%'auto';       % = 'auto'  or   = [-1 1]
tomobaraxis = 'auto';   % = 'auto'  or   = [-1 1]
initial_guess_offset = 0;  % Guess for offset of axis of rotation in pixels
binning = 2;            % Intended for initial fast positioning of large samples - not fully tested
%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%
tic
switch sign(initial_guess_offset)
    case 0
        sino = squeeze(aligned_diff(slice_num,:,:)); % Create the sinogram
    case -1
        sino = squeeze(aligned_diff(slice_num,1:end+2*initial_guess_offset,:)); % Create the sinogram
    case 1
        sino = squeeze(aligned_diff(slice_num,1+2*initial_guess_offset:end,:)); % Create the sinogram
end

if (binning ~=0)&&(binning ~=1)
    display(['Using binning on sinogram = ' num2str(binning)]);
    sinoaux = sino(1:binning:end-binning+1,:);
    for ii = 2:binning
        sinoaux = sinoaux + sino(ii:binning:end-binning+ii,:);
    end
    sino = sinoaux/binning;
end

tomogram_prevalign = tomo.iradonfast_v3(sino, theta-0.01, 'linear', 'derivative', filter_type, size(sino,1), freq_scale);   % Calculate slice

figure(1);
imagesc(sino); 
colormap bone; 
colorbar;
if ~strcmpi(baraxis,'auto')
    caxis(baraxis)
end
axis xy
title(['Sinogram slice ' num2str(slice_num)])
set(gcf,'Outerposition',[1 607 589 418]);

figure(1000+initial_guess_offset); imagesc(tomogram_prevalign); 
if ~strcmp(lower(tomobaraxis),'auto')
    caxis(tomobaraxis)
end
st_title = sprintf('Tomogram slice %d \n initial_guess_offset = %d', slice_num, initial_guess_offset);
title(st_title,'interpreter','none'); colormap bone(256); 
axis equal tight xy; 
colorbar; 
set(gcf,'Outerposition',[1 48 591 559]);

toc


%% Improve alignment with tomographic consistency
%%%%%%%%%%%%%%%%%%%%%%%%%
%%% Edit this section %%%
%%%%%%%%%%%%%%%%%%%%%%%%%
pixtol = 0.1;
disp = 2;
paramsalign.expshift = true;        
paramsalign.interpmeth = 'linear';  % 'sinc' or 'linear'
paramsalign.filtertomo = 0.1;       % frequency cutoff
paramsalign.cliplow  = [];          % Clip air threshold
paramsalign.cliphigh = [0e-3];     % Clip on sample threshold
paramsalign.binning = binning;      % Binning
%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%
tic
deltaslice = zeros(1,size(sino,2), 'single')+initial_guess_offset; % Create the sinogram;
    
sino = squeeze(aligned_diff(slice_num,:,:)); % Create the sinogram

figure(200)
clf
colormap bone
set(gcf,'Outerposition',[1 359 568 665])
figure(200)
clf
colormap bone
set(gcf,'Outerposition',[1 359 568 665])
figure(201)
clf
colormap bone
set(gcf,'Outerposition',[1 48 568 311])
figure(203)
clf
colormap jet
set(gcf,'Outerposition',[569 359 687 665])
figure(2)
clf
set(gcf,'Outerposition',[569 48 687 311])


[deltaslice,sino_align] =  tomo.alignslice_filt_v2(sino,theta-0.01,deltaslice,pixtol,disp,paramsalign);
% sino_align = sino;
toc
%% Here you can add further refinement with different parameters
% paramsalign.binning = binning;
% paramsalign.filtertomo = 0.4;       % frequency cutoff
% paramsalign.cliphigh = [-0.7e-3];%
% [deltaslice,sino_align] =  tomo.alignslice_filt_v2(sino,theta-0.01,deltaslice,pixtol,disp,paramsalign);

%% Automatic alignment (leave uncommented for online tomo)
paramsalign.cliphigh = [0];
tomo_align_binning = [2 1] 

tic   
for temp_bin = tomo_align_binning(1)
    paramsalign.binning = temp_bin;
    for temp_filter = [0.4:0.2:1]
        paramsalign.filtertomo = temp_filter;       % frequency cutoff
        [deltaslice] =  tomo.alignslice_filt_v2(sino,theta-0.01,deltaslice,pixtol,disp,paramsalign);
    end
end

%paramsalign.cliphigh = [0.0001];
if num_proj>300
    for temp_bin = tomo_align_binning(2:end)
        paramsalign.binning = temp_bin;
        for temp_filter = [0.5:0.1:1]
            paramsalign.filtertomo = temp_filter;       
            [deltaslice, sino_align] =  tomo.alignslice_filt_v2(sino,theta-0.01,deltaslice,pixtol,disp,paramsalign);
        end
    end
end
toc

if ~debug()
figure(200);
filename = fullfile(subdir_tomo,sprintf('%sx_align_1.png',online_tomo_fn_out));
fprintf('saving %s\n',filename);
print('-dpng','-r300',filename);
figure(201);
filename = fullfile(subdir_tomo,sprintf('%sx_align_2.png',online_tomo_fn_out));
fprintf('saving %s\n',filename);
print('-dpng','-r300',filename);
figure(203);
filename = fullfile(subdir_tomo,sprintf('%sx_align_3.png',online_tomo_fn_out));
fprintf('saving %s\n',filename);
print('-dpng','-r300',filename);
figure(2);
filename = fullfile(subdir_tomo,sprintf('%sx_align_4.png',online_tomo_fn_out));
fprintf('saving %s\n',filename);
print('-dpng','-r300',filename); 
end

%% Tomo slice
tomogram_align = tomo.iradonfast_v3(sino_align, theta-0.01, 'linear', 'derivative', filter_type, size(sino_align,1), freq_scale);   % Calculate slice

figure(3);
if par.online_tomo
    figure(2000); subplot(2,3,1);
end
imagesc(sino_align); 
colormap bone; 
colorbar;
if ~strcmpi(baraxis,'auto') && (~par.online_tomo)
    caxis(baraxis)
end
axis xy
title(['Sinogram slice ' num2str(slice_num)])
set(gcf,'Outerposition',[590 607 589 418]);

figure(2000); 
if par.online_tomo
    fig_2000_3 = subplot(2,3,3);
end
imagesc(tomogram_align); 
if ~strcmp(lower(tomobaraxis),'auto') && (~par.online_tomo)
    caxis(tomobaraxis)
end
title(['Tomogram slice ' num2str(slice_num)]); colormap bone(256); 
axis equal tight xy; 
colorbar; 
if par.online_tomo
    colormap(fig_2000_3,jet); 
end
caxis([min(tomogram_align(:))*0.95, 0e-3])

    
if par.online_tomo && ~debug()
    figure(2000);         
    temp_datetime = datestr(datetime);
    tomo_time = toc(tomo_tic);
    st_title = sprintf('Tomogram slice %d\n S#%05d~%05d (%d projections) \n theta = [%.2f, %.2f], now %.2f \n load+align = %.0f mins \n Updated on %s',...
    slice_num, par.scanstomo(1), max(par.scanstomo), length(par.scanstomo), min(theta), max(theta), theta(end), tomo_time/60, temp_datetime);
    title(st_title);
    set(gcf,'Outerposition',[1 1 1280 1080]);

    subplot(2,3,2)
    Nmodes = 2; 
    [U,S,V] = svd(medfilt1(massdery,15,'truncate',2)); 
    plot( V(:,[2:Nmodes+1]) * S(2:Nmodes+1,2:Nmodes+1) ); grid on
    title(sprintf('PCA decomposition \n shows changes in the sample'))
    axis tight
    xlabel('Aprox projection number')
    
    figure(2000)
    filename = fullfile(subdir_tomo,sprintf('%s_tomo.png',p.run_name));
    fprintf('saving %s\n',filename);
    print('-dpng','-r300',filename);

    figure(2000)
    filename = fullfile(subdir_tomo,sprintf('lastest_tomo_slice.fig'));
    fprintf('saving %s\n',filename);
    savefig(filename);    
    
    %fn_delta = fullfile(subdir_tomo,sprintf('delta.mat'));
    %save(fn_delta, 'deltastack','deltaslice');
end


%% Tomographic consistency on multiple slices
%%%%%%%%%%%%%%%%%%%%%%%%%
%%% Edit this section %%%
%%%%%%%%%%%%%%%%%%%%%%%%%
slices = slice_num+[-10:10];        % Choose the slices
disp = 0;
%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%
tic
clear deltaxrefine;
parfor ii = slices
    display(['Aligning slice ' num2str(ii)])
    sino = squeeze(aligned_diff(ii,:,:)); % Create the sinogram
    [deltaaux alignedaux] =  tomo.alignslice_filt_v2(sino,theta-0.01,deltaslice,pixtol,disp,paramsalign);
    deltaxrefine(ii,:) = deltaaux;
end
    
deltaxrefine2 = deltaxrefine(slices,:);
deltaxrefineav = mean(deltaxrefine2,1);

figure(1);
imagesc(deltaxrefine2);
xlabel(['Projection number'])
ylabel(['Slice number'])
title(['Displacements in x'])
%caxis([-5 5])
figure(3);
plot(deltaxrefineav)
title(['Average displacements in x'])
xlabel(['Projection number'])

toc
%% Shift projections
%%%%%%%%%%%%%%%%%%%%%%%%%
%%% Edit this section %%%
%%%%%%%%%%%%%%%%%%%%%%%%%
% interpmeth = 'sinc';          % Interpolation method for subpixel shift
% %%%%%%%%%%%%%%%%%%%%%%%%%
% %%%%%%%%%%%%%%%%%%%%%%%%%
tic
aligned_diff_2 = angle(utils.imshift_fft(exp(1i*aligned_diff),-deltaxrefineav,0));
toc

%% [:)] Show aligned derivatives
%%%%%%%%%%%%%%%%%%%%%%%%%
%%% Edit this section %%%
%%%%%%%%%%%%%%%%%%%%%%%%%
show_derivatives = (par.verbose_level > 0); 
par.baraxis = 'auto';       % = 'auto'  or   = [-1 1]
par.showsorted = 1;      % sort the projections by angles 
par.makemovie = 1;
par.movie_range = [];
par.movie_filename = [par.output_folder 'aligned_diff_2.avi']; % for movie output
par.fps=5;
%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%
if show_derivatives
    tomo.show_projections(aligned_diff_2, theta, par,'fps',par.fps)
    if par.makemovie && ~debug()
        if isempty(par.movie_range)
            par.movie_range = 1:size(aligned,3);
        end
        save_movie(aligned_diff(:,:,par.movie_range), par.movie_filename,'bone')
    end
end
%% [:D] Save aligned derivatives
%%%%%%%%%%%%%%%%%%%%%%%%%
%%% Edit this section %%%
%%%%%%%%%%%%%%%%%%%%%%%%%
savedata = par.save_temp_data;
%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%
tic
% If you saved before you may clear
if savedata && ~debug()
    utils.savefast_safe([par.output_folder 'aligned_derivative_projections.mat'], 'aligned_diff_2','deltastack','deltaxrefineav','slice_num','slices','deltaslice','scans','par', 'theta','num_proj','rx','ry','valx','roiy','cut');
end
toc

%% Clear memory 
if par.save_memory
    clear aligned_diff 
end

%% Show tomo on a single slice - Choose tomography parameters
%%%%%%%%%%%%%%%%%%%%%%%%%
%%% Edit this section %%%
%%%%%%%%%%%%%%%%%%%%%%%%%
slice_vis = slice_num;        % Choose the slice to visualize
filter_type = 'Ram-Lak';   % Filter to use for FBP
freq_scale = 1;       % Frequency cutoff
baraxis = [-0.2 0.2];   % = 'auto'  or   = [-1 1]
tomobaraxis = 'auto';   % = 'auto'  or   = [-1 1]
show_slice = ~par.online_tomo;
%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%
if show_slice
    tic
    sino = squeeze(aligned_diff_2(slice_vis,:,:)); % Create the sinogram
    tomogram_slice =tomo.iradonfast_v3(sino, theta, 'linear', 'derivative', filter_type, size(aligned_diff_2,2), freq_scale);   % Calculate slice

    figure(1);
    imagesc(sino); 
    colormap bone; 
    colorbar;
    if ~strcmpi(baraxis,'auto')
        caxis(baraxis)
    end
    axis xy
    title(['Sinogram slice ' num2str(slice_vis)])
    set(gcf,'Outerposition',[1 607 589 418]);

    figure(round(freq_scale*20)); imagesc(tomogram_slice); 
    if ~strcmp(lower(tomobaraxis),'auto')
        caxis(tomobaraxis)
    end
    title(['Tomogram slice ' num2str(slice_vis)]); colormap bone(256); 
    axis equal tight xy; 
    colorbar; 
    set(gcf,'Outerposition',[1 48 591 559]);

    toc
end

%% Full tomogram
%%%%%%%%%%%%%%%%%%%%%%%%%
%%% Edit this section %%%
%%%%%%%%%%%%%%%%%%%%%%%%%
show_while_recons = true;   % Show slices as they are reconstructed
baraxis = [-0.2 0.2];       % Sinogram bar   = 'auto'  or   = [-1 1]
tomobaraxis = 'auto';       % = 'auto'  or   = [-1 1]
usecircle = true;           % Use circle to mask out the corners of the tomogram
%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%
tic

if usecircle
    tomo_size = size(aligned_diff_2,2);
    xt = [-tomo_size/2:tomo_size/2-1];
    [Xt Yt] = meshgrid(xt,xt);
    circulo = 1-radtap(Xt,Yt,10,tomo_size/2-3);
else
    circulo = [];
end

size_tom = size(aligned_diff_2,2);
%%%%%%%%%%%%%%%%%%%%%%%%
% Do the following for parfor when working on the DaaS cluster:
% poolobj = gcp('nocreate');
% if isempty(poolobj) || poolobj.NumWorkers < par.Nworkers
%     delete(poolobj);
%     parpool(par.Nworkers);
% end
%%%%%%%%%%%%%%%%%%%%%%%%%%

% Check RAM
utils.check_available_memory(); 

if gpuDeviceCount
    % use newer GPU version 
    tomogram_delta = tomo.iradonfast_v3_split(aligned_diff_2, theta, 'linear', 'derivative', filter_type, size_tom, freq_scale);   % Calculate slice
    
    if usecircle
        tomogram_delta = tomogram_delta.*circulo;
    end
else
    tomogram_delta = zeros(tomo_size, tomo_size, size(aligned_diff_2,1), 'single');    % Initialise 3D volume
    % old CPU based version 
    parfor ii = 1:size(aligned_diff_2,1)
    %for ii = 1:size(aligned_diff_2,1)
        display(['Slice ' num2str(ii)])
        sino = squeeze(aligned_diff_2(ii,:,:)); % Create the sinogram
        tomogram_delta(:,:,ii) =tomo.iradonfast_v3(sino, theta, 'linear', 'derivative', filter_type, size_tom, freq_scale);   % Calculate slice
        if usecircle
            tomogram_delta(:,:,ii) = tomogram_delta(:,:,ii).*circulo;
        end
        if show_while_recons
            figure(1);
            imagesc(sino);
            colormap bone;
            colorbar;
            if ~strcmpi(baraxis,'auto')
                caxis(baraxis)
            end
            axis xy
            title(['Sinogram slice ' num2str(ii)])
            set(gcf,'Outerposition',[1 607 589 418]);

            figure(2); imagesc(tomogram_delta(:,:,ii));
            if ~strcmp(lower(tomobaraxis),'auto')
                caxis(tomobaraxis)
            end
            title(['Tomogram slice ' num2str(ii)]); colormap bone(256);
            axis equal tight xy;
            colorbar;
            set(gcf,'Outerposition',[1 48 591 559]);
            drawnow
        end
    end
end
%%% Convert to delta values %%%
tomogram_delta = -tomogram_delta*par.factor;
% store size of the tomogram 
Npix = size(tomogram_delta,1); 

%% [:)] Display reconstructed volume by slices
%%%%%%%%%%%%%%%%%%%%%%%%%
%%% Edit this section %%%
%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%% Either show standard cuts %%%%
show_standard_cuts = par.online_tomo;   % Shows three cuts through the middle of the tomogram. Then overrides par.displayaxis and par.displayslice
%%%%% Or choose here which slice and direction to show with the following %%%
par.displayaxis = 3;        % 1 coronal, 2 sagital, 3 axial
par.displayslice = [];      % Many slices supported e.g [100:200]. If left empty the center slice is shown, 
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
par.animatedslices = false; % All slices are shown ignoring those in displayslice
par.average_slices = false; % If true, it ignores animatedslices
% Display options
par.tomobaraxis = 'auto';  % = 'auto'  or e.g.  = [-1 1]. For auto it picks max and min from whole tomogram
par.scale = 'delta';            % = 'phase', 'delta' [index] or 'edensity' [electron/A^3]
par.colormapchoice = 'bone';    % Choose the colormap
par.realaxis = true;            % = true to show result with real axes units, = false for pixels
par.reverse_contrast = false;    % Reverse grayscale
% Scale bar options (only shown if realaxis = true)
par.bar_length= 5e-6;           % Lenght of scalebar (only used if realaxis = true)
par.bar_height= 0.5e-6;           % Height of scalebar (only used if realaxis = true)
par.bar_start_point=[0.8 0.8]*1e-6;    % (x, y) starting point of the scale bar (from corner) in meters
par.bar_color = 'w';            % Color of scalebar
% Other
par.windowautopos = false;      % If true the window position and size is changed, can be useful as false when ready to render a movie
par.makemovie = false;          % = true for rendering an AVI
par.writesnapshots = 0;      % Makes jpeg and eps snaphots. It overwrites snapshots. If many slices are chosen it writes only the last one.
par.pausetime = 0.1;            % Pause time for animation or multiple slices
% other 
par.freq_scale = freq_scale; 
par.filter_type = filter_type; 
%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%

% Coronal, first index
% Sagital, second index
% Axial,   third index
if ~show_standard_cuts
    figure(1); clf
    tomo.show_tomogram_cuts(tomogram_delta, par.scanstomo, par)
else 
    figure(2000); 
    subplot(2,3,4); 
    par.displayaxis = 1; par.displayslice = round(size(tomogram_delta,2)/2);
    tomo.show_tomogram_cuts(tomogram_delta, par.scanstomo, par); title(sprintf('[e/A^3] Coronal slice %d',par.displayslice))
    subplot(2,3,5); 
    par.displayaxis = 2; par.displayslice = round(size(tomogram_delta,2)/2);
    tomo.show_tomogram_cuts(tomogram_delta, par.scanstomo, par); title(sprintf('Sagital slice %d',par.displayslice))
    subplot(2,3,6); 
    par.displayaxis = 3; par.displayslice = round(size(tomogram_delta,3)/2);
    tomo.show_tomogram_cuts(tomogram_delta, par.scanstomo, par); title(sprintf('Axial slice %d',par.displayslice))
end

%% upload a slice to the sample database
%% Attention!!!measurement id from the printout. if that is wrong, it will overwrite files! be careful!
if ~debug()
measurement_id = 9999; 
filename_database = '~/Data10/analysis_tomo/tomo_S00036_to_S01145_run_2/S00037_S01145_edensity_freqscl_1.00_axis_1_slice_378.png';
unix_cmd = sprintf('/work/sls/spec/local/XOMNY/bin/upload/upload_tomography_slice.sh %d %s',measurement_id, filename_database);
fprintf('%s\n',unix_cmd)
unix(unix_cmd);
end
%% [:D] Save Tomogram (delta)
%%%%%%%%%%%%%%%%%%%%%%%%%
%%% Edit this section %%%
%%%%%%%%%%%%%%%%%%%%%%%%%
savedata = par.save_final_data;
%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%
% Note saving tomogram in delta (index of refraction)
% To get phase tomogram   = -tomgram_delta/factor
% To get electron density tomogram in [e/A^3]   = tomogram_delta*factor_edensity
% If you saved before you may clear 

tic
% If you saved before you may clear
if savedata && ~debug()
    saveprojfile = sprintf('%stomogram_delta_S%05d_S%05d_%s_freqscl_%0.2f.mat',par.output_folder,...
        par.scanstomo(1),par.scanstomo(end),filter_type,freq_scale);
    utils.savefast_safe(saveprojfile,'par','tomogram_delta','circulo','filter_type','freq_scale','scans', 'theta','num_proj','rx','ry');%, par.force_overwrite);
end
toc


%% [:D] Save Tiff files for 3D visualization with external program
%%%%%%%%%%%%%%%%%%%%%%%%%
%%% Edit this section %%%
%%%%%%%%%%%%%%%%%%%%%%%%%
% par.filter_type = ''; 
% par.freq_scale = 0; 
par.save_as_stack = false; 
par.tiff_compression = 'none';
par.tiff_subfolder_name = ['TIFF_delta_' par.filter_type '_freqscl_' sprintf('%0.2f',par.freq_scale)];
par.name_prefix = 'tomo_delta';
%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%
if par.save_final_data && ~debug()
    tomo.save_as_tiff(tomogram_delta, par)
end
% Conversion formula
% im_delta_from_tiff = im_tiff*(high_cutoff-low_cutoff)/(2^16-1) + low_cutoff;
% im_edensity_from_tiff = im_delta_from_tiff*factor_edensity;


%% 2D FSC on a single slice - Warning: can be a pesimistic estimate if angular resolution is low
%%%%%%%%%%%%%%%%%%%%%%%%%
%%% Edit this section %%%
%%%%%%%%%%%%%%%%%%%%%%%%%
compute_2D_FSC = true;
slice_FSC = 47;
show_figures = (par.verbose_level > 0);
thickring = 10;
SNRt = 0.2071; %1/2 bit  % Threshold curve for FSC
radial_apod_width = Npix/20;         % width of the apodization smoothing 
par.out_fn = sprintf('%sFSC_2D_slice%d_radapod%.0f_1.png', par.output_folder, slice_FSC, radial_apod_width);
par.windowautopos = true;
par.clear_figure = true;
%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%
tic
display(['FSC on slice ' num2str(slice_FSC)])
[~, ind] = sort(theta);
ind1   = ind(1:2:end);
ind2   = ind(2:2:end);

sino1 = squeeze(aligned_diff_2(slice_FSC,:,ind1));
sino2 = squeeze(aligned_diff_2(slice_FSC,:,ind2));
theta1 = theta(ind1); 
theta2 = theta(ind2);
Wsino = size(aligned_diff_2,2);
tomoslice1 =tomo.iradonfast_v3(sino1, theta1, 'linear', 'derivative', filter_type, Wsino, freq_scale);   % Calculate slice
tomoslice2 =tomo.iradonfast_v3(sino2, theta2, 'linear', 'derivative', filter_type, Wsino, freq_scale);   % Calculate slice

tomoslice1 = apply_3D_apodization(tomoslice1, radial_apod_width, []);
tomoslice2 = apply_3D_apodization(tomoslice2, radial_apod_width, []);

if show_figures
    figure(2);
    imagesc(tomoslice1);
    title(['Tomogram slice ' num2str(slice_FSC)]); colormap bone(256);
    axis equal tight xy;
    colorbar;
    set(gcf,'Outerposition',[1 48 591 559]);
    
    figure(3);
    imagesc(tomoslice2);
    title(['Tomogram slice ' num2str(slice_FSC)]); colormap bone(256);
    axis equal tight xy;
    colorbar;
    set(gcf,'Outerposition',[590 48 591 559]);
end

if debug() && isfield(par,'out_fn')
    par = rmfield(par, 'out_fn'); 
end
[res_2d FSC T r n] = utils.fourier_shell_corr_3D_2(tomoslice1,tomoslice2, par,'SNRt', SNRt, 'thickring', thickring);


%% Choose region for FSC
%%%%%%%%%%%%%%%%%%%%%%%%%
%%% Edit this section %%%
%%%%%%%%%%%%%%%%%%%%%%%%%
limsyFSC = 10:size(tomogram_delta)-10;
%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%

showslice = round(Npix/2);
sliceview = squeeze(tomogram_delta(showslice,:,:))';

figure(1)
imagesc(sliceview)
axis xy equal tight
hold on
plot([1 size(sliceview,2)],[1 1]*limsyFSC(1),'r')
plot([1 size(sliceview,2)],[1 1]*limsyFSC(end),'r')
hold off

%% Clear memory 
if par.save_memory
    clear tomogram_delta 
end


%% Compute subtomograms for FSC
%%%%%%%%%%%%%%%%%%%%%%%%%
%%% Edit this section %%%
%%%%%%%%%%%%%%%%%%%%%%%%%
show_figures = (par.verbose_level > 1);
showslice = round(Npix/2);
%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%

tic
if gpuDeviceCount
    % use newer GPU version 
    tomogram1 = tomo.iradonfast_v3_split(aligned_diff_2(limsyFSC,:,ind1), theta1-0.01, 'linear', 'derivative', filter_type, size_tom, freq_scale);   % Calculate slice
    tomogram2 = tomo.iradonfast_v3_split(aligned_diff_2(limsyFSC,:,ind2), theta2-0.01, 'linear', 'derivative', filter_type, size_tom, freq_scale);   % Calculate slice
else
    % old CPU based version 
    tic
    tomogram1 = zeros([size(aligned_diff_2,2) size(aligned_diff_2,2) numel(limsyFSC)], 'single');
    tomogram2 = tomogram1;

    size_tom=size(aligned_diff_2,2);
    %%%%%%%%%%%%%%%%%%%%%%%%
    % Do the following for parfor when working on the DaaS cluster:
    % poolobj = gcp('nocreate');
    % if isempty(poolobj) || poolobj.NumWorkers < par.Nworkers
    %     delete(poolobj);
    %     parpool(par.Nworkers);
    % end
    %%%%%%%%%%%%%%%%%%%%%%%%%%
    % parfor ii = 1:length(limsyFSC)
    for ii = 1:length(limsyFSC)
        display(['Slice ' num2str(limsyFSC(ii))])
        sino1 = squeeze(aligned_diff_2(limsyFSC(ii),:,ind1));
        sino2 = squeeze(aligned_diff_2(limsyFSC(ii),:,ind2));
        tomogram1(:,:,ii) =tomo.iradonfast_v2(sino1, theta1, 'linear', 'derivative', filter_type, size_tom, freq_scale);   % Calculate slice
        tomogram2(:,:,ii) =tomo.iradonfast_v2(sino2, theta2, 'linear', 'derivative', filter_type, size_tom, freq_scale);   % Calculate slice
    end
    if show_figures
        sliceview1 = squeeze(tomogram1(showslice,:,:))';
        sliceview2 = squeeze(tomogram2(showslice,:,:))';

        figure(2)
        imagesc(sliceview1)
        axis xy equal tight

        figure(3)
        imagesc(sliceview2)
        axis xy equal tight
    end
end

toc
%% Filter subtomos for FSC - Warning: This can take a while, watch out for memory if you have large datasets
%%%%%%%%%%%%%%%%%%%%%%%%%
%%% Edit this section %%%
%%%%%%%%%%%%%%%%%%%%%%%%%
show_figures = true;
rad_apod = round(size(tomogram1,1)/10);      % Radial apodization
radial_apod_width = round(size(tomogram1,1)/10);         % width of the apodization smoothing 
axial_apod = round(size(tomogram1,3)/10);                     % Axial apodization
showslice = round(size(tomogram1,1)/2);
%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%
tic

auxtomo1 = apply_3D_apodization(tomogram1,rad_apod, axial_apod, radial_apod_width); 
auxtomo2 = apply_3D_apodization(tomogram2,rad_apod, axial_apod, radial_apod_width); 


if show_figures
    sliceview1 = squeeze(auxtomo1(showslice,:,:))';
    sliceview2 = squeeze(auxtomo2(showslice,:,:))';
    
    figure(2)
    set(gcf,'Outerposition',[1 582 600 418]);
    imagesc(sliceview1)
    axis xy equal tight
    colormap bone
    title('Subtomogram 1')
    
    
    figure(3)
    set(gcf,'Outerposition',[601 582 600 418]);
    imagesc(sliceview2)
    axis xy equal tight
    title('Subtomogram 2')
    colormap bone
    
    
    figure(4)
    set(gcf,'Outerposition',[1 26 600 556]);
    imagesc(auxtomo1(:,:,round(size(auxtomo1,3)/2)));
    axis xy equal tight
    title('Subtomogram 1')
    colormap bone
    
    figure(5)
    set(gcf,'Outerposition',[601 26 600 556]);
    imagesc(auxtomo2(:,:,round(size(auxtomo2,3)/2)));
    axis xy equal tight
    title('Subtomogram 2')
    colormap bone
end


toc
%% Compute FSC and save results - Warning: can be a pesimistic estimate if angular resolution is low
% Warning(2): This can take a while, watch out for memory if you have large datasets
%%%%%%%%%%%%%%%%%%%%%%%%%
%%% Edit this section %%%
%%%%%%%%%%%%%%%%%%%%%%%%%
savedata = par.save_final_data;
par.SNRt = 0.2071; %1/2 bit  % Threshold curve for FSC
par.out_fn = sprintf('%sFSC_3D_limy%d-%d_radapod%.0f_1.png', par.output_folder, limsyFSC(1), limsyFSC(end), rad_apod);
par.windowautopos = true;
par.clear_figure = true;
par.auto_binning = true; 
par.thickring = 4;    % in combination with auto_binning it will make rings with thickness = "par.thickring" along the smallest dimension
%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%
if debug() && isfield(par,'out_fn')
    par = rmfield(par, 'out_fn'); 
end
[res_3d FSC T r n] = utils.fourier_shell_corr_3D_2(auxtomo1,auxtomo2,par);

if savedata && ~debug()
     saveprojfile = sprintf('%sFSC_S%05d_S%05d_%s_freqscl_%0.2f_radapod%d_limy%d-%d.mat',par.output_folder,...
        par.scanstomo(1),par.scanstomo(end),filter_type,freq_scale,rad_apod, limsyFSC(1), limsyFSC(end));
    utils.savefast_safe(saveprojfile,'par','FSC','T','r','n','rad_apod','axial_apod','circulo','filter_type','freq_scale','scans','par', 'theta','num_proj','rx','ry', 'limsyFSC');%, par.force_overwrite);
end


%% Clear memory 
if par.save_memory
    clear auxtomo1  auxtomo2 tomogram1 tomogram2 aligned_diff_2 
end

%% Take screenshot 
figure(2000); 
title(sprintf('Axial slice %d-%d\n 2D resol ~%.1f nm\n 3D resol ~%.1f nm',par.displayslice(1), par.displayslice(end), res_2d(1), res_3d(1)))

if ~debug()
% Save to online/ptycho/
filename = fullfile(subdir_online, sprintf('%s_tomo.png',p.run_name));
fprintf('saving %s\n',filename);
print('-dpng','-r300',filename);
    
% Save to subdir_tomo
filename = fullfile(subdir_tomo, sprintf('%s_tomo_S%05d.png',p.run_name, par.scanstomo(end)));
fprintf('saving %s\n',filename);
print('-dpng','-r300',filename);
end



par.save_final_data = true; 


% Then go back and run again the sections for saving 
% [:D] sections are necessary (ie, alinged, tomgram, and TIFF)
% [:)] sections are optional (eg, unwrapped, movie for aligned)
%
% If you are unhappy about the unwrapping
% [**] sections: force remove ramp, force unwrap

%% Determine calibration for sphere for software auto-alignment tomography

param_calib.savedata = 1;
param_calib.surface_calib_file = [par.base_path sprintf('matlab/tomo/position_calibration_2017_S%05d_S%05d.mat',par.scanstomo(1),par.scanstomo(end))];
param_calib.get_auto_calibration = par.get_auto_calibration;
param_calib.pixel_size = par.pixel_size;
param_calib.theta = theta;
param_calib.scans = par.scanstomo;
param_calib.output_folder = par.output_folder;

if ~debug()
tomo.give_calibration(obj_interf_pos_x, obj_interf_pos_y , deltastack, deltaslice,  theta, scans, param_calib);
end

%%% Determine calibration for sphere for measuring with auto-alignment tomography
% par.savedata = 0;
%par.surface_calib_file = [par.base_path 'matlab/tomo/position_calibration_20180301.mat'];

%tomo.give_calibration(obj_interf_pos_x, obj_interf_pos_y , deltastack, deltaslice, theta, scans, par);


%% Dose estimation
%%%%%%%%%%%%%%%%%%%%%%%%%
%%% Edit this section %%%
%%%%%%%%%%%%%%%%%%%%%%%%
%num_proj = 1;       % Number of tomographic projections
par.mu = 1/(451*1e-6); % 1/attenuation_length   in 1/m   (for CH2 @6.2keV)
                     % 1/(152.7*1e-6) for zeolite Na2Al2Si3O102H4O with 2 g/cm3 density at 6.2 keV
par.rho = 1000;          % Density in kg/m^3
par.setup_transmission = 0.55; % Intensity transmission of sample 
                        % (e.g. air path after the sample, windows, He, detector efficiency)
                        % 0.943 for 700 cm He gas at 760 Torr and 295 K @ 6.2 keV
                        % 0.780 for 10 cm air at 760 Torr and 295 K @ 6.2 keV
                        % 0.976 for 13 micron Kapton (polymide) with 1.43
                        % g/cm3 @ 6.2 keV
                        % 0.841 for 7 micron muskovite mica
                        % (KAl3Si3O11.8H1.8F0.2) with 2.76 g/cm3 @ 6.2 keV
                        % 0.914 for 5 cm of air at 6.2 keV 750 Torr 295 K
                        % 0.55 for 300 micron of mylar C10H8O4 with density 1.38 g/cm3 at 6.2 keV
par.overhead = 0.0;         % Extra dose during movement overhead, only applicable 
                        %      if shutter is not closed between exposures
                        %      (meastime/exptime - 1)
                       
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%  
if ~debug()
scan_0 = par.scanstomo(1);   % Choose first                       
ptycho_filename = find_ptycho_filename(par.analysis_path,scan_0,par.fileprefix,par.filesuffix, par.file_extension);
if iscell(ptycho_filename); ptycho_filename = ptycho_filename{end}; end
ptycho_recon = io.load_ptycho_recons(ptycho_filename);

data_filename = dir([par.analysis_path,sprintf('S%5.5d',scan_0), '/*_data_*.', par.file_extension]); % change it to .mat if needed
assert(~isempty(data_filename), 'Klaus new bug: data are saved only in every second scan number')
data_filename = fullfile(data_filename(1).folder, '/',data_filename(1).name);
[ ptycho_data.data, ptycho_data.fmask] = io.load_prepared_data( data_filename, true,[],true); 



par.num_proj = num_proj; 
par.scan_number = ii;

utils.dose_calc(ptycho_recon,ptycho_data,par);

% Clear memory 
clear ptycho_recon ptycho_data 
end


%% [AMPLITUDE] Tomography with amplitude
% If started from here, we need to load reconstructed_projections.mat and 
% aligned_derivative_projections.mat

if ~exist('stack_object', 'var')
   stack_object = load([par.output_folder, '/reconstructed_projections.mat']); 
   stack_object = stack_object.stack_object_r + 1i*stack_object.stack_object_i; 
end

stack_amp = abs(stack_object);

%% Clear memory 
if par.save_memory
    clear stack_object
end


%% Display reconstructed amplitudes
%%%%%%%%%%%%%%%%%%%%%%%%%
%%% Edit this section %%%
%%%%%%%%%%%%%%%%%%%%%%%%%
show_amplitudes = par.verbose_level > 0; 
par.baraxis = 'auto';       % = 'auto'  or   = [-1 1]
par.showsorted = 0;      % sort the projections by angles 
par.makemovie = 0;  
%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%
if show_amplitudes
    tomo.show_projections(stack_amp, theta, par)
end

%% Load mask for ramp removal or run again the GUI for doing this
if ~debug()
load(sprintf('%s/mask_auto_S%05d.mat',par.output_folder, min(scans)));
end

%% Display reconstructed amplitudes with mask
%%%%%%%%%%%%%%%%%%%%%%%%%
%%% Edit this section %%%
%%%%%%%%%%%%%%%%%%%%%%%%%
par.baraxis = 'auto';       % = 'auto'  or   = [-1 1]
par.showsorted = 0;      % sort the projections by angles
par.makemovie = 0;  
%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%
if show_amplitudes
    figure(1); clf
    tomo.show_projections( bsxfun(@times, stack_amp, mask) , theta, par)
end

%% Correct amplitude factor (use same mask as for phase ramp corection)
if debug()
    mask(:,[1,end],:) = true; 
end

vals = math.sum2(bsxfun(@times, mask,stack_amp))./ math.sum2(mask);
corrected_stack_amp = bsxfun(@rdivide, stack_amp, vals); 


figure(1);
plot(squeeze(vals)); grid on; title('correction for amplitude factor'); axis tight 


%% Display reconstructed amplitudes with corrected amplitude factor
%%%%%%%%%%%%%%%%%%%%%%%%%
%%% Edit this section %%%
%%%%%%%%%%%%%%%%%%%%%%%%%
par.baraxis = 'auto';       % = 'auto'  or   = [-1 1]
par.showsorted = 0;      % sort the projections by angles 
par.makemovie = 0;  
%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%
if show_amplitudes
    figure(1); clf
    tomo.show_projections( corrected_stack_amp , theta, par)
end


%% Clear memory 
if par.save_memory
    clear stack_amp
end


%% Shift the projections using previous alignment found for the phase tomogram
tic
aligned_stack_amp = utils.imshift_fft(corrected_stack_amp, -deltastack(2,:)-deltaxrefineav(1,:), -deltastack(1,:));
aligned_stack_amp = abs(aligned_stack_amp);
toc

%% Display aligned amplitudes
%%%%%%%%%%%%%%%%%%%%%%%%%
%%% Edit this section %%%
%%%%%%%%%%%%%%%%%%%%%%%%%
par.baraxis = 'auto';       % = 'auto'  or   = [-1 1]
par.showsorted = 1;      % sort the projections by angles 
par.makemovie = 0;  
%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%
if show_amplitudes
    figure(1); clf
    tomo.show_projections( aligned_stack_amp , theta, par)
end

%% Clear memory 
if par.save_memory
    clear corrected_stack_amp
end

%% Select region for tomography equal to that used for phase projections
%%%%%%%%%%%%%%%%%%%%%%%%%
%%% Edit this section %%%
%%%%%%%%%%%%%%%%%%%%%%%%%
par.baraxis = [0.5 1.2];%'auto';       % = 'auto'  or   = [-1 1]
par.makemovie = 0;  
%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%
roix=[1+valx+cut size(aligned_stack_amp,2)-valx-cut];
if show_amplitudes
    tomo.show_projections( aligned_stack_amp , theta, par, 'rectangle_pos', [roix, roiy])
end
%% See region of interest
%%%%%%%%%%%%%%%%%%%%%%%%%
%%% Edit this section %%%
%%%%%%%%%%%%%%%%%%%%%%%%%
par.baraxis = 'auto';       % = 'auto'  or   = [-1 1]
par.makemovie = 0;  
%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%
aligned_amp_sel=aligned_stack_amp(roiy(1):roiy(2),roix(1):roix(2),:);
if show_amplitudes
    figure(1); clf
    tomo.show_projections( aligned_amp_sel , theta, par)
end

%% Save aligned amplitudes
%%%%%%%%%%%%%%%%%%%%%%%%%
%%% Edit this section %%%
%%%%%%%%%%%%%%%%%%%%%%%%%
savedata = par.save_temp_data;
%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%
% If you saved before you may clear 
if savedata && ~debug()
    utils.savefast_safe([par.output_folder 'aligned_amplitude_projections.mat'],'par','aligned_amp_sel','deltastack','deltaxrefineav','slice_num','slices','deltaslice','scans','theta','num_proj','rx','ry','valx','roiy','cut', par.force_overwrite);
end

%% Clear memory 
if par.save_memory
    clear aligned_stack_amp
end



%% %% Visualize some sinograms and tomograms
%%%%%%%%%%%%%%%%%%%%%%%%%
%%% Edit this section %%%
%%%%%%%%%%%%%%%%%%%%%%%%%
slice_num = 20;        % Choose the slice
filter_type_amp = 'Hann';   % Filter to use for FBP
freq_scale_amp = 0.5;       % Frequency cutoff
baraxis = [-0.5 0.5];%'auto';       % = 'auto'  or   = [-1 1]
tomobaraxis = 'auto';   % = 'auto'  or   = [-1 1]
initial_guess_offset = 0;  % Guess for offset of axis of rotation in pixels
binning = 0;            % Intended for initial fast positioning of large samples - not fully tested
show_amp_tomo = par.verbose_level > 0; 
%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%

switch sign(initial_guess_offset)
    case 0
        sino = squeeze(log(aligned_amp_sel(slice_num,:,:))); % Create the sinogram
    case -1
        sino = squeeze(log(aligned_amp_sel(slice_num,1:end+2*initial_guess_offset,:))); % Create the sinogram
    case 1
        sino = squeeze(log(aligned_amp_sel(slice_num,1+2*initial_guess_offset:end,:))); % Create the sinogram
end

if (binning ~=0)&&(binning ~=1)
    sino = utils.binning_2D(sino)*binning; 
%     display(['Using binning on sinogram = ' num2str(binning)]);
%     sinoaux = sino(1:binning:end-binning,:);
%     for ii = 2:binning
%         sinoaux = sinoaux + sino(ii:binning:end-binning+ii,:);
%     end
%     sino = sinoaux/binning;
end

tomogram_prevalign = -par.factor*tomo.iradonfast_v3(sino, theta-0.01, 'linear', filter_type_amp, size(sino,1), freq_scale_amp);   % Calculate slice

if show_amp_tomo
    figure(1);
    imagesc(sino); 
    colormap bone; 
    colorbar;
    if ~strcmpi(baraxis,'auto')
        caxis(baraxis)
    end
    axis xy
    title(['Sinogram slice ' num2str(slice_num)])
    set(gcf,'Outerposition',[1 607 589 418]);

    figure(1002); imagesc(tomogram_prevalign); 
    if ~strcmp(lower(tomobaraxis),'auto')
        caxis(tomobaraxis)
    end
    title(['Tomogram amplitude slice ' num2str(slice_num)]); colormap bone(256); 
    axis equal tight xy; 
    colorbar; 
    set(gcf,'Outerposition',[1 48 591 559]);
end


%% Full tomogram with amplitude
%%%%%%%%%%%%%%%%%%%%%%%%%
%%% Edit this section %%%
%%%%%%%%%%%%%%%%%%%%%%%%%
show_while_recons = true;   % Show slices as they are reconstructed
baraxis = [-0.2 0.2];       % Sinogram bar   = 'auto'  or   = [-1 1]
tomobaraxis = 'auto';       % = 'auto'  or   = [-1 1]
usecircle = true;           % Use circle to mask out the corners of the tomogram
rad_apod = 20;
%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%
tomo_size=size(aligned_amp_sel,2);


if gpuDeviceCount
    % use newer GPU version 
    tomogram_beta = tomo.iradonfast_v3_split(log(aligned_amp_sel), theta-0.01,'linear', filter_type_amp,  size_tom, freq_scale_amp);   % Calculate slice
    if usecircle
        tomogram_beta = apply_3D_apodization(tomogram_beta, rad_apod);
    end
else
    % old CPU based version 

    tomogram_beta = zeros(tomo_size, tomo_size, size(aligned_amp_sel,1), 'single');    % Initialise 3D volume


    size_tom=size(aligned_amp_sel,2);

    for ii = 1:size(aligned_amp_sel,1)
        display(['Slice ' num2str(ii)])
        sino = squeeze(log(aligned_amp_sel(ii,:,:))); % Create the sinogram
        tomogram_beta(:,:,ii) =tomo.iradonfast_v2(sino, theta-0.01, 'linear', filter_type_amp, size_tom, freq_scale_amp);   % Calculate slice

        if usecircle
            tomogram_beta(:,:,ii) = apply_3D_apodization(tomogram_beta(:,:,ii), size(tomogram_beta,1)-size(sino,2));
        end
        if show_while_recons
            figure(1);
            imagesc(sino);
            colormap bone;
            colorbar;
            if ~strcmpi(baraxis,'auto')
                caxis(baraxis)
            end
            axis xy
            title(['Sinogram slice ' num2str(ii)])
            set(gcf,'Outerposition',[1 607 589 418]);

            figure(2); imagesc(tomogram_beta(:,:,ii));
            if ~strcmp(lower(tomobaraxis),'auto')
                caxis(tomobaraxis)
            end
            title(['Tomogram amplitude slice ' num2str(ii)]); colormap bone(256);
            axis equal tight xy;
            colorbar;
            set(gcf,'Outerposition',[1 48 591 559]);
            drawnow
        end
    end
end

tomogram_beta = -tomogram_beta*par.factor;

%% Display reconstructed volume by slices
%%%%%%%%%%%%%%%%%%%%%%%%%
%%% Edit this section %%%
%%%%%%%%%%%%%%%%%%%%%%%%%
par.displayaxis = 1;            % 1 coronal, 2 sagital, 3 axial
par.displayslice = [];         % Many slices supported e.g [100:200]. If left empty the center slice is shown, 
par.animatedslices = false;     % All slices are shown ignoring those in displayslice
par.average_slices = false;    % If true, it ignores animatedslices
% Display options
par.tomobaraxis = 'auto';       % = 'auto'  or e.g.  = [-1 1]. For auto it picks max and min from whole tomogram
par.scale = 'beta';            % = 'phase', 'delta' [index] or 'edensity' [electron/A^3]
par.colormapchoice = 'bone';    % Choose the colormap
par.realaxis = true;            % = true to show result with real axes units, = false for pixels
par.reverse_contrast = false;    % Reverse grayscale
% Scale bar options (only shown if realaxis = true)
par.bar_length= 2e-6;           % Lenght of scalebar (only used if realaxis = true)
par.bar_height= 0.2e-6;           % Height of scalebar (only used if realaxis = true)
par.bar_start_point=[0.8 0.2]*1e-6;    % (x, y) starting point of the scale bar (from corner) in meters
par.bar_color = 'w';            % Color of scalebar
% Other
par.windowautopos = true;      % If true the window position and size is changed, can be useful as false when ready to render a movie
par.makemovie = false;          % = true for rendering an AVI
par.writesnapshots = true;      % Makes jpeg and eps snaphots. It overwrites snapshots. If many slices are chosen it writes only the last one.
par.pausetime = 0.1;            % Pause time for animation or multiple slices
%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%

% Coronal, first index
% Sagital, second index
% Axial,   third index

par_amp = par; % temp solution
par_amp.filter_type = filter_type_amp;
par_amp.freq_scale = freq_scale_amp;
tomo.show_tomogram_cuts(tomogram_beta, par.scanstomo, par_amp)


%% Save Tomogram (beta)
%%%%%%%%%%%%%%%%%%%%%%%%%
%%% Edit this section %%%
%%%%%%%%%%%%%%%%%%%%%%%%%
savedata = par.save_final_data;
%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%
% Note saving tomogram in delta (index of refraction)
% To get phase tomogram   = -tomgram_delta/factor
% To get electron density tomogram in [e/A^3]   = tomogram_delta*factor_edensity
% If you saved before you may clear 

if savedata && ~debug()
    saveprojfile = sprintf('%stomogram_beta_S%05d_S%05d_%s_freqscl_%0.2f.mat',par.output_folder,...
        par.scanstomo(1),par.scanstomo(end),filter_type_amp,freq_scale_amp);
    utils.savefast_safe(saveprojfile,'par','tomogram_beta','circulo','filter_type_amp','freq_scale_amp','scans','num_proj','rx','ry', par.force_overwrite);
end


%% Save Tiff files for 3D visualization with external program
%%%%%%%%%%%%%%%%%%%%%%%%%
%%% Edit this section %%%
%%%%%%%%%%%%%%%%%%%%%%%%%
par.save_as_stack = false; 
par.tiff_compression = 'none';
par.tiff_subfolder_name = ['TIFF_beta_' filter_type_amp '_freqscl_' sprintf('%0.2f',freq_scale_amp)];
%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%
if savedata && ~debug()
    % Note that par does not have filter_type_amp nor freq_scale_amp..
    % temp solution:
    par_amp = par;
    par_amp.filter_type = filter_type_amp;
    par_amp.freq_scale = freq_scale_amp;
    tomo.save_as_tiff(tomogram_beta, par_amp)
end


%% Clear memory 
if par.save_memory
    clear tomogram_beta aligned_amp_sel
end



%*-----------------------------------------------------------------------*
%|                                                                       |
%|  Except where otherwise noted, this work is licensed under a          |
%|  Creative Commons Attribution-NonCommercial-ShareAlike 4.0            |
%|  International (CC BY-NC-SA 4.0) license.                             |
%|                                                                       |
%|  Copyright (c) 2017 by Paul Scherrer Institute (http://www.psi.ch)    |
%|                                                                       |
%|       Author: CXS group, PSI                                          |
%*-----------------------------------------------------------------------*
% You may use this code with the following provisions:
%
% If the code is fully or partially redistributed, or rewritten in another
%   computing language this notice should be included in the redistribution.
%
% If this code, or subfunctions or parts of it, is used for research in a 
%   publication or if it is fully or partially rewritten for another 
%   computing language the authors and institution should be acknowledged 
%   in written form in the publication: “Data processing was carried out 
%   using the “cSAXS matlab package” developed by the CXS group,
%   Paul Scherrer Institut, Switzerland.” 
%   Variations on the latter text can be incorporated upon discussion with 
%   the CXS group if needed to more specifically reflect the use of the package 
%   for the published work.
%
% A publication that focuses on describing features, or parameters, that
%    are already existing in the code should be first discussed with the
%    authors.
%   
% This code and subroutines are part of a continuous development, they 
%    are provided “as they are” without guarantees or liability on part
%    of PSI or the authors. It is the user responsibility to ensure its 
%    proper use and the correctness of the results.
